const whitespace = require('is-whitespace-character');

const S_LINK_LOC = 'mailto:|(ref:)?file:(?!https?://)|(ref:)?https?://';
const S_LINK_LA0 = `^(?:${S_LINK_LOC})`;

const R_LINK_LOC = new RegExp(S_LINK_LOC, 'g');
// Флаг i для поддержки регистронезависимых протоколов в начале строки
// https://st.yandex-team.ru/WIKI-13901
const R_LINK_LA0 = new RegExp(S_LINK_LA0, 'i');

const { decode: htmlEntitiesDecode } = require('../utils/html-entities');

const C_BRACKET_OPEN = '[';
const C_BRACKET_CLOSE = ']';
const C_PAREN_OPEN = '(';
const C_PAREN_CLOSE = ')';
const C_LT = '<';
const C_AT_SIGN = '@';

const REF_MARK = 'ref:';
const MAILTO_PROTOCOL = 'mailto:';

const TL_HAS_POPEN_SPACE = 0x01;
const TL_HAS_BOPEN_SPACE = 0x02;
const TL_HAS_SPACE_PCLOSE = 0x04;
const TL_HAS_SPACE_BCLOSE = 0x08;

function url(eat, value, silent) {
    const self = this;
    let subvalue;
    let content;
    let character;
    let index;
    let position;
    let match;
    let length;
    let queue;
    let parenCount;
    let nextCharacter;
    let exit;
    let tlBingo;
    let tlPos;
    let ref;

    match = R_LINK_LA0.exec(value);

    if (!match) {
        return false;
    }

    subvalue = match[0];
    ref = Boolean(match[1] || match[2]);

    index = subvalue.length;
    length = value.length;
    queue = '';
    parenCount = 0;

    tlBingo = 0;
    tlPos = [];

    while (index < length) {
        character = value.charAt(index);

        if ((parenCount === 0 && whitespace(character)) || character === C_LT) {
            break;
        }

        if (
            character === '.' ||
            character === ',' ||
            character === ':' ||
            character === ';' ||
            character === '"' ||
            character === '\'' ||
            (parenCount <= 0 && (
                character === ')' ||
                character === ']'
            ))
        ) {
            nextCharacter = value.charAt(index + 1);

            if (!nextCharacter || parenCount === 0 && whitespace(nextCharacter)) {
                break;
            }
        }

        const isParen = character === C_PAREN_OPEN || character === C_PAREN_CLOSE;
        // ( or [
        if (character === C_PAREN_OPEN || character === C_BRACKET_OPEN) {
            parenCount ++;

            if (whitespace(value.charAt(index + 1))) {
                tlBingo = tlBingo | (isParen ? TL_HAS_POPEN_SPACE : TL_HAS_BOPEN_SPACE);
                tlPos[isParen ? 2 : 0] = Math.min(index, tlPos[isParen ? 2 : 0] || Infinity);
            }
        }

        // ) or ]
        if (character === C_PAREN_CLOSE || character === C_BRACKET_CLOSE) {
            parenCount --;

            if (whitespace(value.charAt(index - 1))) {
                tlBingo = tlBingo | (character === C_PAREN_CLOSE ? TL_HAS_SPACE_PCLOSE : TL_HAS_SPACE_BCLOSE);
                tlPos[isParen ? 3 : 1] = Math.max(index, tlPos[isParen ? 3 : 1] || 0);
            }

            if (parenCount < 0) {
                break;
            }
        }

        queue += character;
        index ++;
    }

    if (!queue) {
        return;
    }

    subvalue += queue;
    content = subvalue;

    if (match[0].toLowerCase() === MAILTO_PROTOCOL) {
        position = queue.indexOf(C_AT_SIGN);

        if (position === - 1 || position === length - 1) {
            return;
        }

        content = content.substr(MAILTO_PROTOCOL.length);
    }

    /* istanbul ignore if - never used (yet) */
    if (silent) {
        return true;
    }

    const offset = ref ? REF_MARK.length : 0;

    exit = self.enterLink();

    const now = eat.now();
    now.offset += offset;
    now.column += offset;

    const children = [];

    self.decode(content.slice(offset), now, value => {
        children.push({ type: 'text', value });
    });

    exit();

    return eat(subvalue)({
        type: 'link',
        title: null,
        url: htmlEntitiesDecode(subvalue.slice(offset)),
        ref: ref || undefined,
        children,
    });
}

url.locator = (value, fromIndex) => {
    R_LINK_LOC.lastIndex = fromIndex;

    const m = R_LINK_LOC.exec(value);

    if (m) {
        return m.index;
    }

    return - 1;
};

module.exports = url;
