class StateNode {
    constructor({ type, inline }) {
        // Тип узла
        this.type = type;
        // Строчный или блоковый
        this.inline = inline;
        // Любые атрибуты
        this.attributes = {};
        // Начало открывающего разделителя
        this.openingInitialIndex = -1;
        // Конец открывающего разделителя
        this.openingFollowingIndex = -1;
        // Начало контента
        this.innerFirstIndex = -1;
        // Начало закрывающего разделителя
        this.closingInitialIndex = -1;
        // Конец закрывающего разделителя
        this.closingFollowingIndex = -1;
        // Начало контента после узла
        this.outerFirstIndex = -1;
        // Дочерние узлы
        this.children = [];
    }
}

class StackItem {
    constructor(container, outerNode, stateNode) {
        // Искомый контейнер
        this.container = container;
        // Родительский узел
        this.outerNode = outerNode;
        // Текущий узел
        this.stateNode = stateNode;
        // Свертки текущего состояния
        this.innerPops = [];
    }
}

function makeUnknownNode_({ inline }, openingIndex, closingIndex) {
    const stateNode = new StateNode({ type: 'unknown', inline });

    stateNode.openingInitialIndex =
        stateNode.openingFollowingIndex =
            stateNode.innerFirstIndex = openingIndex;

    stateNode.closingInitialIndex =
        stateNode.closingFollowingIndex =
            stateNode.outerFirstIndex = closingIndex;

    return stateNode;
}

function foldUnknownNode_(stateNode, closingIndex) {
    const { children } = stateNode;
    const { length } = children;
    let openingIndex = -1;

    if (length === 0) {
        openingIndex = stateNode.innerFirstIndex;
    } else {
        openingIndex = children[length - 1].outerFirstIndex;
    }

    if (!(closingIndex > openingIndex)) {
        return;
    }

    if (length === 0 || children[length - 1].type !== 'unknown') {
        children.push(makeUnknownNode_(stateNode, openingIndex, closingIndex));
        return;
    }

    // Можно смержить ноды.
    // Не обрабатываю кейсы, когда ноды пересекаются,
    // такого не бывает by-design и это не покрыть тестом
    children[length - 1].closingInitialIndex = closingIndex;
    children[length - 1].closingFollowingIndex = closingIndex;
    children[length - 1].outerFirstIndex = closingIndex;
}

function isClosingPossible_(value, container, stateNode) {
    const { innerFirstIndex } = stateNode;

    const l = value.length;
    // Shallow clone
    const cloneNode = { ...stateNode };

    for (let i = innerFirstIndex; i < l; i += 1) {
        container.matchClosing(value, i, cloneNode);

        if (cloneNode.closingInitialIndex < 0) {
            continue;
        }

        return true;
    }

    return false;
}

function foldStateNode_(outerNode, stateNode) {
    // Заворачиваем контент предшествующий свернутому узлу в unknown узел
    foldUnknownNode_(outerNode, stateNode.openingInitialIndex);

    // Перед закрытием контейнера добавляем в его конец unknown узел
    foldUnknownNode_(stateNode, stateNode.closingInitialIndex);

    // Добавляем в родительский узел свернутый распаршенный узел
    outerNode.children.push(stateNode);
}

function canBorrowPops_(prevState, headState) {
    const { innerPops } = headState;

    // Проверка, может ли состояние получить все свертки
    const innerPopsLength = innerPops.length;

    if (innerPopsLength === 0) {
        return true;
    }

    const { container: { possibleChildren } } = prevState;
    const containersLength = possibleChildren.length;

    if (containersLength === 0) {
        return false;
    }

    const set = new Set();

    for (let i = 0; i < containersLength; i += 1) {
        set.add(possibleChildren[i]);
    }

    for (let i = 0; i < innerPopsLength; i += 1) {
        const { container } = innerPops[i];

        if (set.has(container)) {
            continue;
        }

        return false;
    }

    return true;
}

function hasTheSameStateBehind_(stack) {
    const lastIndex = stack.length - 1;
    const { container } = stack[lastIndex];
    let prevIndex = lastIndex - 1;

    while (prevIndex > 0) {
        if (stack[prevIndex].container === container) {
            return true;
        }

        prevIndex -= 1;
    }

    return false;
}

function enterSpacing_(stack, value, currIndex) {
    const { container, stateNode } = stack[stack.length - 1];

    container.enterSpacing(value, currIndex, stateNode);
}

function checkSpacing_(stack, value, currIndex) {
    const { container, stateNode } = stack[stack.length - 1];

    if (container.checkSpacing(value, currIndex, stateNode)) {
        return currIndex + 1;
    }

    return -1;
}

function matchClosing_(stack, value, currIndex) {
    const { container, stateNode, outerNode } = stack[stack.length - 1];

    // Пытаемся закрыть текущий контейнер
    container.matchClosing(value, currIndex, stateNode);

    if (stateNode.closingInitialIndex < 0) {
        return -1;
    }

    // Нашли закрывающий разделитель, делаем свертку
    foldStateNode_(outerNode, stateNode);

    const rightState = stack.pop();

    if (stack.length > 0) {
        stack[stack.length - 1].innerPops.push(rightState);
    }

    // Продолжаем поиск с индекса сразу после найденного узла
    return stateNode.outerFirstIndex;
}

function matchOpening_(stack, value, currIndex, skipEnter) {
    const { container, stateNode } = stack[stack.length - 1];

    const { possibleChildren } = container;
    const containersCount = possibleChildren.length;

    // Пытаемся открыть какой-нибудь контейнер внутри текущего
    for (let i = 0; i < containersCount; i += 1) {
        const container = possibleChildren[i];

        if (skipEnter[currIndex] && skipEnter[currIndex].has(container)) {
            // Индекс запрещен для перехода в это состояние
            continue;
        }

        // Создаю вероятный внутренний узел
        const innerNode = new StateNode(container);

        container.matchOpening(value, currIndex, innerNode);

        if (innerNode.openingInitialIndex < 0) {
            continue;
        }

        // PERF: Заранее проверяю, есть ли вообще шансы закрыть этот узел
        if (!isClosingPossible_(value, container, innerNode)) {
            continue;
        }

        //  Кладем на стек соответствующее состояние
        stack.push(new StackItem(container, stateNode, innerNode));

        return innerNode.innerFirstIndex;
    }

    return -1;
}

function cutWrongState_(stack, skipEnter) {
    // Удаляем последнее состояние, потому что оно неправильное
    const wrongState = stack.pop();

    // PERF: Запоминаем, что в текущем, новом состоянии
    // больше не нужно заходить в неправильное состояние
    const { stateNode: { openingInitialIndex }, container } = wrongState;

    if (!skipEnter[openingInitialIndex]) {
        skipEnter[openingInitialIndex] = new Set();
    }

    skipEnter[openingInitialIndex].add(container);

    const { stateNode } = wrongState;

    //  Удаляем всех детей неправильного состояния
    stateNode.children.length = 0;

    // Берем узел верхнего состояния, он точно не закрыт
    // Продолжаем сканировать, не заходя в неправильное состояние
    return stateNode.openingInitialIndex;
}

function parseTheContainerAt(container, value, fromIndex) {
    const stateNode = new StateNode(container);

    container.matchOpening(value, fromIndex, stateNode);

    if (stateNode.openingInitialIndex < 0) {
        // Ранний выход, чтобы сделать код менее вложенным
        return stateNode;
    }

    // PERF: Заранее проверяю, есть ли шанс закрыть узел
    if (!isClosingPossible_(value, container, stateNode)) {
        return stateNode;
    }

    // Индексы, по которым не нужно заходить в состояние
    const skipEnter = Object.create(null);

    const stack = [
        // Пушим корневое состояние в стек
        new StackItem(
            container,
            // Фейковый родительский узел
            new StateNode({ type: '', inline: false }),
            stateNode,
        ),
    ];

    let currIndex = stateNode.innerFirstIndex;
    const valueLength = value.length;

    outer: while (stack.length > 0) {
        // Вызываем хук входа в неопознанный контент
        enterSpacing_(stack, value, currIndex);

        while (currIndex < valueLength) {
            let nextIndex = matchClosing_(stack, value, currIndex);

            if (nextIndex > 0) {
                currIndex = nextIndex;

                continue outer;
            }

            nextIndex = matchOpening_(stack, value, currIndex, skipEnter);

            if (nextIndex > 0) {
                currIndex = nextIndex;

                continue outer;
            }

            // У контейнеров может быть разная логика связанная с перемещением
            // по контенту.
            // Например, строчные MD форматтеры не могут содержать пустую строку,
            // но WOM-строчники прекрасно себя чувствуют в таких ситуациях.
            // Какое-то состояние можно хранить внутри stateNode
            nextIndex = checkSpacing_(stack, value, currIndex);

            if (nextIndex > 0) {
                currIndex = nextIndex;

                continue;
            }

            break;
        }

        // PERF: Есть часть состояний, в которых в любом случае не будет сверток,
        // поэтому их можно сразу удалить
        while (stack.length > 2 && hasTheSameStateBehind_(stack)) {
            currIndex = cutWrongState_(stack, skipEnter);
        }

        const headState = stack[stack.length - 1];

        do {
            // Удаляем неправильные состояния до тех пор,
            // пока предыдущее состояние не сможет
            // позаимствовать свертки текущего
            currIndex = cutWrongState_(stack, skipEnter);
        } while (stack.length > 0 && !canBorrowPops_(stack[stack.length - 1], headState));
    }

    return stateNode;
}

exports.parseTheContainerAt = parseTheContainerAt;
exports.createUnknownNode = makeUnknownNode_;
