const {
    isWhitespaceChar,
    isLineFeedChar,
} = require('./spec');

function isLineOpening(value, index) {
    if (index === 0) {
        return true;
    }

    return isLineFeedChar(value, index - 1);
}

function isLineSpacing(value, index) {
    let prevIndex = index;

    while (!isLineOpening(value, prevIndex)) {
        if (isWhitespaceChar(value, prevIndex - 1)) {
            prevIndex -= 1;
            continue;
        }

        return false;
    }

    return true;
}

function isStickySupported() {
    try {
        RegExp('', 'y');
        return true;
    } catch (_error) {
        /* istanbul ignore next */
        return false;
    }
}

function createNativeStickyMatcher(source) {
    const regExp = new RegExp(source, 'y');

    return (value, lastIndex) => {
        regExp.lastIndex = lastIndex;

        return regExp.exec(value);
    };
}

function createPolyfillStickyMatcher(source) {
    const regExp = new RegExp(source, 'g');

    return (value, lastIndex) => {
        regExp.lastIndex = lastIndex;

        const match = regExp.exec(value);

        if (match !== null) {
            if (lastIndex + match[0].length === regExp.lastIndex) {
                return match;
            }
        }

        return null;
    };
}

const createStickyMatcher = isStickySupported() ?
    createNativeStickyMatcher :
    /* istanbul ignore next */
    createPolyfillStickyMatcher;

function skipLineFollowingSpacing(value, fromIndex) {
    let nextIndex = fromIndex;
    const l = value.length;

    while (nextIndex < l) {
        if (isWhitespaceChar(value, nextIndex)) {
            if (isLineFeedChar(value, nextIndex)) {
                // Пробельный символ оказался переносом строки

                return nextIndex;
            }

            // Если это не перенос строки, то дальше перебираем пробелы
            nextIndex += 1;

            continue;
        }

        // Если мы встретили НЕ пробел, то результат отрицательный
        return -1;
    }

    // Цикл был пройден до конца, значит, строка,
    // начиная с fromIndex состоит из 0 или более пробелов.
    // Это можно считать концом строки

    return nextIndex;
}

function skipLineRest(value, fromIndex) {
    let nextIndex = fromIndex;
    const l = value.length;

    while (nextIndex < l) {
        if (isLineFeedChar(value, nextIndex)) {
            // Достигнут конец строки
            return nextIndex;
        }

        nextIndex += 1;
    }

    return nextIndex;
}

// Для блочных форматтеров
function trimPrecedingBlankLine(value, { children }) {
    // Некоторые блоки, например ячейки,
    // могут начинать свой контент на той же строке
    // нужно обрезать пробелы до начала текста или остановиться сразу после переноса строки
    while (children.length > 0 && children[0].type === 'unknown') {
        const firstChild = children[0];

        if (firstChild.innerFirstIndex < firstChild.closingInitialIndex) {
            const { openingInitialIndex } = firstChild;

            if (isWhitespaceChar(value, openingInitialIndex)) {
                firstChild.openingInitialIndex += 1;
                firstChild.openingFollowingIndex += 1;
                firstChild.innerFirstIndex += 1;

                if (isLineFeedChar(value, openingInitialIndex)) {
                    // Но это был перенос строки. Значит надо остановиться
                    break;
                }

                continue;
            }

            break;
        }

        children.shift();
    }
}

function trimPrecedingSpacing(value, { children }) {
    while (children.length > 0 && children[0].type === 'unknown') {
        const firstChild = children[0];

        if (firstChild.innerFirstIndex < firstChild.closingInitialIndex) {
            if (isWhitespaceChar(value, firstChild.openingInitialIndex)) {
                firstChild.openingInitialIndex += 1;
                firstChild.openingFollowingIndex += 1;
                firstChild.innerFirstIndex += 1;

                continue;
            }

            break;
        }

        children.shift();
    }
}

function trimFollowingSpacing(value, { children }) {
    while (children.length > 0 && children[children.length - 1].type === 'unknown') {
        const lastChild = children[children.length - 1];

        if (lastChild.innerFirstIndex < lastChild.closingInitialIndex) {
            if (isWhitespaceChar(value, lastChild.outerFirstIndex - 1)) {
                lastChild.closingInitialIndex -= 1;
                lastChild.closingFollowingIndex -= 1;
                lastChild.outerFirstIndex -= 1;

                continue;
            }

            break;
        }

        children.pop();
    }
}

// Для строчных форматеров
function trimEqualSpacingAround(value, { children }) {
    if (children.length === 0) {
        return;
    }

    const firstChild = children[0];
    const lastChild = children[children.length - 1];

    if (firstChild.type !== 'unknown' || lastChild.type !== 'unknown') {
        return;
    }

    while (
        // not empty
        firstChild.openingInitialIndex < firstChild.outerFirstIndex &&
        // not empty
        lastChild.openingInitialIndex < lastChild.outerFirstIndex &&
        // the same trailing chars
        value[firstChild.openingInitialIndex] === value[lastChild.outerFirstIndex - 1] &&
        // which is whitespace
        isWhitespaceChar(value, firstChild.openingInitialIndex)
    ) {
        firstChild.openingInitialIndex += 1;
        firstChild.openingFollowingIndex += 1;
        firstChild.innerFirstIndex += 1;

        lastChild.closingInitialIndex -= 1;
        lastChild.closingFollowingIndex -= 1;
        lastChild.outerFirstIndex -= 1;
    }

    if (lastChild.openingInitialIndex === lastChild.outerFirstIndex) {
        // Empty unknown last child
        children.pop();
    }

    if (firstChild.openingInitialIndex === firstChild.outerFirstIndex) {
        // Empty unknown first child
        children.shift();
    }
}

function getNodeInnerText(value, { children }) {
    if (children.length === 0) {
        return '';
    }

    const { openingInitialIndex } = children[0];
    const { outerFirstIndex } = children[children.length - 1];

    return value.slice(openingInitialIndex, outerFirstIndex);
}

function getNodeOuterText(value, { openingInitialIndex, outerFirstIndex }) {
    return value.slice(openingInitialIndex, outerFirstIndex);
}

const parseContainerParams = (() => {
    const matcher = createStickyMatcher(
        '\\s*' +
        // key
        '([^\\s=]+)' +
        // val
        '(?:' +
        // = "foo" | 'bar'
        '(?:\\s*=\\s*(?:\'([^\']*)\'|"([^"]*)")(?=\\s|$))' +
        '|' +
        // = foo
        '(?:\\s*=\\s*?([^\\s]*))' +
        ')?' +
        '\\s*'
    );

    return value => {
        const params = {};
        let lastIndex = 0;

        while (lastIndex < value.length) {
            const match = matcher(value, lastIndex);

            if (match === null) {
                return {};
            }

            params[match[1]] = match[2] || match[3] || match[4] || null;

            lastIndex += match[0].length;
        }

        return params;
    };
})();

const createParamsParser = (openingSource, closingSource) => {
    const R_REGEX_SYNTAX = /[-$()*+.\/?[\\\]^{|}]/g;
    const regEsc = source => source.replace(R_REGEX_SYNTAX, '\\$&');
    const opening = regEsc(openingSource);
    const closing = regEsc(closingSource);

    const matcher = createStickyMatcher(
        `${opening}\\s*([\\w+-/#.]+)([\\s\\S]*?)${closing}`
    );

    return (value, fromIndex) => {
        const match = matcher(value, fromIndex);

        if (match !== null) {
            return {
                format: match[1].toLowerCase(),
                params: match[2],
                offset: fromIndex + match[0].length,
            };
        }

        return {
            format: '',
            params: {},
            offset: fromIndex,
        };
    };
};

const parseWomFormatterHeader = createParamsParser('(', ')');
const parseWomActionHeader = createParamsParser('{{', '}}');

exports.createNativeStickyMatcher = createNativeStickyMatcher;
exports.createPolyfillStickyMatcher = createPolyfillStickyMatcher;

exports.createStickyMatcher = createStickyMatcher;

exports.skipLineFollowingSpacing = skipLineFollowingSpacing;
exports.skipLineRest = skipLineRest;

exports.trimPrecedingBlankLine = trimPrecedingBlankLine;
exports.trimEqualSpacingAround = trimEqualSpacingAround;

exports.trimPrecedingSpacing = trimPrecedingSpacing;
exports.trimFollowingSpacing = trimFollowingSpacing;

exports.getNodeInnerText = getNodeInnerText;
exports.getNodeOuterText = getNodeOuterText;

exports.isLineOpening = isLineOpening;
exports.isLineSpacing = isLineSpacing;

exports.parseContainerParams = parseContainerParams;
exports.parseWomFormatterHeader = parseWomFormatterHeader;
exports.parseWomActionHeader = parseWomActionHeader;
