const {
    isEscaped,
    isWhitespaceChar,
    isLineFeedChar,
} = require('../spec');
const {
    isLineOpening,
    isLineSpacing,
    skipLineFollowingSpacing,
    skipLineRest,
} = require('../skip-blocks-utils');

// Md форматтер кода (block)
const blockMdCode = {
    type: 'code',
    inline: false,
    possibleChildren: [],
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 3) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        if (value[openingInitialIndex] !== '`') {
            // Early return
            return;
        }

        // Невозможно заэкранировать открывающий разделитель блока,
        // потому что перед ним всегда есть \n, либо он в самом начале строки.
        // Поэтому на экранирование не проверяем.
        // Проверка начала строки - ниже
        let openingFollowingIndex = openingInitialIndex + 1;

        while (openingFollowingIndex < l && value[openingFollowingIndex] === '`') {
            openingFollowingIndex += 1;
        }

        const headDelimiterSize = openingFollowingIndex - openingInitialIndex;

        if (headDelimiterSize < 3) {
            return;
        }

        const innerFirstIndex = skipLineRest(value, openingFollowingIndex);
        const closingPos = value.indexOf('`'.repeat(headDelimiterSize), openingFollowingIndex);

        if (closingPos > -1 && closingPos < innerFirstIndex) {
            return;
        }

        if (!isLineOpening(value, openingInitialIndex) || skipLineFollowingSpacing(value, innerFirstIndex) < 0) {
            return;
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = innerFirstIndex;
        stateNode.attributes.format = value
            .slice(openingFollowingIndex, innerFirstIndex)
            .trim().split(/\s+/).shift();
    },
    matchClosing(value, closingInitialIndex, stateNode) {
        const l = value.length;

        if (l - closingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        if (value[closingInitialIndex] !== '`') {
            // Early return
            return;
        }

        // Закрывающий разделитель нельзя заэкранировать
        let closingFollowingIndex = closingInitialIndex + 1;

        while (closingFollowingIndex < l && value[closingFollowingIndex] === '`') {
            closingFollowingIndex += 1;
        }

        const { openingInitialIndex, openingFollowingIndex } = stateNode;

        if (openingFollowingIndex - openingInitialIndex !== closingFollowingIndex - closingInitialIndex) {
            return;
        }

        if (!isLineSpacing(value, closingInitialIndex) || skipLineFollowingSpacing(value, closingFollowingIndex) < 0) {
            return;
        }

        stateNode.closingInitialIndex = closingInitialIndex;
        stateNode.closingFollowingIndex = closingFollowingIndex;
        stateNode.outerFirstIndex = closingFollowingIndex;
    },
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

// Md форматтер кода (inline)
const inlineMdCode = {
    type: 'inlineCode',
    inline: true,
    possibleChildren: [],
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 1) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        if (value[openingInitialIndex] !== '`') {
            // Early return
            return;
        }

        // Открывающий разделитель может быть заэкранирован
        // Перед открывающим разделителем идет любой другой незаэкранированный символ
        if (isEscaped(value, openingInitialIndex) || (value[openingInitialIndex - 1] === '`' && !isEscaped(value, openingInitialIndex - 1))) {
            return;
        }

        let openingFollowingIndex = openingInitialIndex + 1;

        while (l > openingFollowingIndex && value[openingFollowingIndex] === '`') {
            openingFollowingIndex += 1;
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = openingFollowingIndex;
    },
    matchClosing(value, closingInitialIndex, stateNode) {
        const l = value.length;

        if (value[closingInitialIndex] !== '`') {
            // Early return
            return;
        }

        // Закрывающий разделитель нельзя заэкранировать
        // Перед закрывающим разделителем идет любой другой символ
        if (value[closingInitialIndex - 1] === '`') {
            return;
        }

        let closingFollowingIndex = closingInitialIndex;

        while (closingFollowingIndex < l && value[closingFollowingIndex] === '`') {
            closingFollowingIndex += 1;
        }

        const { openingInitialIndex, openingFollowingIndex } = stateNode;

        if (openingFollowingIndex - openingInitialIndex !== closingFollowingIndex - closingInitialIndex) {
            return;
        }

        stateNode.closingInitialIndex = closingInitialIndex;
        stateNode.closingFollowingIndex = closingFollowingIndex;
        stateNode.outerFirstIndex = closingFollowingIndex;
    },
    enterSpacing(value, spacingCurrentIndex, stateNode) {
        stateNode.lineFeedCount = 0;
    },
    checkSpacing(value, spacingCurrentIndex, stateNode) {
        if (isLineFeedChar(value, spacingCurrentIndex)) {
            stateNode.lineFeedCount += 1;

            if (stateNode.lineFeedCount > 1) {
                return false;
            }
        } else if (!isWhitespaceChar(value, spacingCurrentIndex)) {
            stateNode.lineFeedCount = 0;
        }

        return true;
    },
};

exports.blockMdCode = blockMdCode;
exports.inlineMdCode = inlineMdCode;
