const { expect } = require('chai');
const { parseTheContainerAt } = require('../parse-container-at');

const {
    blockMdCode,
    inlineMdCode,
} = require('../skip-blocks');

const createLA0ContainerParser = container =>
    (value, fromIndex) =>
        parseTheContainerAt(container, value, fromIndex);

const parseBlockMdCode = createLA0ContainerParser(blockMdCode);
const parseInlineMdCode = createLA0ContainerParser(inlineMdCode);

function traverse(node, fn) {
    fn(node);

    for (const childNode of node.children) {
        traverse(childNode, fn);
    }
}

function rmExtraProps(root) {
    traverse(root, node => {
        delete node.lineFeedCount;
    });
}

function runSamples(fn, samples) {
    for (const [args, expected] of samples) {
        it(JSON.stringify([...args]), () => {
            const node = fn(...args);

            rmExtraProps(node);

            expect(node).to.deep.equal(expected);
        });
    }
}

// 0 &&
describe('mdCode(block)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['``', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['xxx', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['x```\n1\n```', 1],
                {
                    type: 'code',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['```1```', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['``\n1\n```', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['```\n1\nx```', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: { format: '' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['```\n1\n```x', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: { format: '' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['```1```', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['```js\n1\n```', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: {
                        format: 'js',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 5,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 5,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['```js foo\n1\n```', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: {
                        format: 'js',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 9,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 9,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['```\n1\n```', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: { format: '' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 9,
                    outerFirstIndex: 9,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 3,
                            innerFirstIndex: 3,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 6,
                            outerFirstIndex: 6,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['```\n\n1\n``', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: { format: '' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['```\n1\n``', 0],
                {
                    type: 'code',
                    inline: false,
                    attributes: { format: '' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
        ],
    );

    runSamples(parseBlockMdCode, samples);
});

// 0 &&
describe('mdCode(inline)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['x', 1],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['\\`1`', 1],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['`\n\n1\n\n`', 0],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 1,
                    innerFirstIndex: 1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['`\nx\n1\nx\n`', 0],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 1,
                    innerFirstIndex: 1,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 9,
                    outerFirstIndex: 9,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 1,
                            openingFollowingIndex: 1,
                            innerFirstIndex: 1,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['`\n \n1\n \n`', 0],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 1,
                    innerFirstIndex: 1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['`1`', 0],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 1,
                    innerFirstIndex: 1,
                    closingInitialIndex: 2,
                    closingFollowingIndex: 3,
                    outerFirstIndex: 3,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 1,
                            openingFollowingIndex: 1,
                            innerFirstIndex: 1,
                            closingInitialIndex: 2,
                            closingFollowingIndex: 2,
                            outerFirstIndex: 2,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['`1``', 0],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 1,
                    innerFirstIndex: 1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['``1`', 0],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['\\``1`', 2],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 2,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 4,
                    closingFollowingIndex: 5,
                    outerFirstIndex: 5,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 3,
                            innerFirstIndex: 3,
                            closingInitialIndex: 4,
                            closingFollowingIndex: 4,
                            outerFirstIndex: 4,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['1`', 0],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['```1```', 0],
                {
                    type: 'inlineCode',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 4,
                    closingFollowingIndex: 7,
                    outerFirstIndex: 7,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 3,
                            innerFirstIndex: 3,
                            closingInitialIndex: 4,
                            closingFollowingIndex: 4,
                            outerFirstIndex: 4,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    runSamples(parseInlineMdCode, samples);
});
