const { expect } = require('chai');
const { parseTheContainerAt } = require('../parse-container-at');

const {
    blockWomBlockquote,
    inlineWomBlockquote,
} = require('../skip-blocks');

const createLA0ContainerParser = container =>
    (value, fromIndex) =>
        parseTheContainerAt(container, value, fromIndex);

const parseBlockWomBlockquote = createLA0ContainerParser(blockWomBlockquote);
const parseInlineWomBlockquote = createLA0ContainerParser(inlineWomBlockquote);

function traverse(node, fn) {
    fn(node);

    for (const childNode of node.children) {
        traverse(childNode, fn);
    }
}

function rmExtraProps(root) {
    traverse(root, node => {
        delete node.lineFeedCount;
    });
}

function runSamples(fn, samples) {
    for (const [args, expected] of samples) {
        it(JSON.stringify([...args]), () => {
            const node = fn(...args);

            rmExtraProps(node);

            expect(node).to.deep.equal(expected);
        });
    }
}

// 0 &&
describe('womBlockquote(block)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['<', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<[\n1\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 5,
                    closingFollowingIndex: 7,
                    outerFirstIndex: 7,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 5,
                            outerFirstIndex: 5,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n<[\n1\n]>\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 11,
                    closingFollowingIndex: 13,
                    outerFirstIndex: 13,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womBlockquote',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 8,
                                    outerFirstIndex: 8,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n<[\n1\n]>\n]>\n<[\n<[\n1\n]>\n]>\n', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 11,
                    closingFollowingIndex: 13,
                    outerFirstIndex: 13,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womBlockquote',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 8,
                                    outerFirstIndex: 8,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n%%\n1\n%%\n]>\n', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 11,
                    closingFollowingIndex: 13,
                    outerFirstIndex: 13,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 8,
                                    outerFirstIndex: 8,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n%%\n1\n%%\n]>\n<[\n%%\n1\n%%\n]>\n', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 11,
                    closingFollowingIndex: 13,
                    outerFirstIndex: 13,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 8,
                                    outerFirstIndex: 8,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n%%%\n1\n%%%\n]>\n', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 13,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 6,
                                    innerFirstIndex: 6,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n%%%\n1\n%%%\n]>\n<[\n%%%\n1\n%%%\n]>\n', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 13,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 6,
                                    innerFirstIndex: 6,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n<[\n1\n]>', 3],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 3,
                    openingFollowingIndex: 5,
                    innerFirstIndex: 5,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 10,
                    outerFirstIndex: 10,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 5,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n<[\n<[\n<[\n<[\n]>', 12],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 12,
                    openingFollowingIndex: 14,
                    innerFirstIndex: 14,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            openingFollowingIndex: 14,
                            innerFirstIndex: 14,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n{[\n]>\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 8,
                    outerFirstIndex: 8,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 6,
                            outerFirstIndex: 6,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n{[\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 8,
                    outerFirstIndex: 8,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 6,
                            outerFirstIndex: 6,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    // 0 &&
    samples.push(
        ...[
            [
                ['<[\n%%\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 8,
                    outerFirstIndex: 8,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 6,
                            outerFirstIndex: 6,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    // 0 &&
    samples.push(
        ...[
            [
                ['<[\n%%\n]>\n%%\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 14,
                    outerFirstIndex: 14,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 11,
                            openingFollowingIndex: 11,
                            innerFirstIndex: 11,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    // 0 &&
    samples.push(
        ...[
            [
                ['<[\n{[\n]>\n]}\n]>\n', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 14,
                    outerFirstIndex: 14,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womBlock',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 11,
                            openingFollowingIndex: 11,
                            innerFirstIndex: 11,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n<{\n]>\n}>\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 14,
                    outerFirstIndex: 14,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 11,
                            openingFollowingIndex: 11,
                            innerFirstIndex: 11,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n%%\n]>\n%%\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 14,
                    outerFirstIndex: 14,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 11,
                            openingFollowingIndex: 11,
                            innerFirstIndex: 11,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[\n%%(md)\n]>\n%%\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 16,
                    closingFollowingIndex: 18,
                    outerFirstIndex: 18,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womMarkdown',
                            inline: false,
                            attributes: {
                                format: 'md',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 9,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 13,
                                    closingFollowingIndex: 13,
                                    outerFirstIndex: 13,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 15,
                            openingFollowingIndex: 15,
                            innerFirstIndex: 15,
                            closingInitialIndex: 16,
                            closingFollowingIndex: 16,
                            outerFirstIndex: 16,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['<[', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<[]>', 0],
                {
                    type: 'womBlockquote',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
        ],
    );

    runSamples(parseBlockWomBlockquote, samples);
});

// 0 &&
describe('womBlockquote(inline)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['<', 0],
                {
                    type: 'womBlockquote',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<[ ~<[ ]> ]>', 0],
                {
                    type: 'womBlockquote',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 7,
                    closingFollowingIndex: 9,
                    outerFirstIndex: 9,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womEscape',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 4,
                            innerFirstIndex: 4,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 6,
                            outerFirstIndex: 6,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: true,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 4,
                                    innerFirstIndex: 4,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 6,
                                    outerFirstIndex: 6,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<', 0],
                {
                    type: 'womBlockquote',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<[]', 0],
                {
                    type: 'womBlockquote',
                    inline: true,
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    attributes: {},
                    children: [],
                },
            ],
            [
                ['<[\n]>', 0],
                {
                    type: 'womBlockquote',
                    inline: true,
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 3,
                    closingFollowingIndex: 5,
                    outerFirstIndex: 5,
                    attributes: {},
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<[]>', 0],
                {
                    type: 'womBlockquote',
                    inline: true,
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 2,
                    closingFollowingIndex: 4,
                    outerFirstIndex: 4,
                    attributes: {},
                    children: [],
                },
            ],
            [
                ['<!]>', 0],
                {
                    type: 'womBlockquote',
                    inline: true,
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    attributes: {},
                    children: [],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['<[x<[x<[x]>x<[x<[x]>', 0],
                {
                    type: 'womBlockquote',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
        ],
    );

    runSamples(parseInlineWomBlockquote, samples);
});
