const { isEscaped } = require('../spec');
const { findMatchBetween } = require('../parse-container-utils');
const {
    defaultBlockChildren,
    defaultInlineChildren,
} = require('./possible-children');

// Wom кат (block)
const blockWomCut = {
    type: 'womCut',
    inline: false,
    possibleChildren: defaultBlockChildren,
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Невозможно заэкранировать открывающий разделитель блока,
        // потому что перед ним всегда есть \n, либо он в самом начале строки.
        // Поэтому на экранирование не проверяем.
        // Проверка начала строки - ниже
        if (!(value[openingInitialIndex] === '<' && value[openingInitialIndex + 1] === '{')) {
            return;
        }

        const openingFollowingIndex = openingInitialIndex + 2;
        // В блочном кате все, что идет до конца строки
        // после открывающего разделителя есть его title
        // Но в этом title может быть любая верстка

        //  Находим конец строки
        let innerFirstIndex = value.indexOf('\n', openingFollowingIndex);

        if (innerFirstIndex < 0) {
            // В любом случае кат не закроется
            return;
        }

        if (findMatchBetween(defaultBlockChildren, value, openingFollowingIndex, innerFirstIndex, i => {
            if (value[i] === '}' && value[i + 1] === '>') {
                return i;
            }

            return -1;
        }) > 0) {
            // Это значит, что кат закрылся на этой же строке
            // Нам это не подходит, потому что блочный кат не может закрыться на этой же строке
            return;
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = innerFirstIndex;
        stateNode.attributes.title = value.slice(openingFollowingIndex, innerFirstIndex);
    },
    matchClosing(value, closingInitialIndex, stateNode) {
        const l = value.length;

        if (l - closingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Закрывающий разделитель можно заэкранировать
        if (!(value[closingInitialIndex] === '}' && value[closingInitialIndex + 1] === '>') || isEscaped(value, closingInitialIndex)) {
            return;
        }

        const closingFollowingIndex = closingInitialIndex + 2;

        stateNode.closingInitialIndex = closingInitialIndex;
        stateNode.closingFollowingIndex = closingFollowingIndex;
        stateNode.outerFirstIndex = closingFollowingIndex;
    },
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

// Wom кат (inline)
const inlineWomCut = {
    type: 'womCut',
    inline: true,
    // Несмотря на то, что кат строчный, в нем могут быть блоки
    possibleChildren: defaultBlockChildren,
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Открывающий разделитель можно заэкранировать
        if (!(value[openingInitialIndex] === '<' && value[openingInitialIndex + 1] === '{') || isEscaped(value, openingInitialIndex)) {
            return;
        }

        const openingFollowingIndex = openingInitialIndex + 2;

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;

        // Сложнее понять что может быть в title у строчного ката
        // 1. Если строчный кат располагается на нескольких строках,
        // то *title - это все до конца строки*
        //  Пример:
        //  <{test
        //  я вполне строчный кат, потому что во мне нет `/(?:\n\s*){2,}/`.
        //  Хотя я также могу быть и блочным, а как меня отличить от блочного, никто не знает
        //  }>
        // 2. Если строчный кат располагается на одной строке, что страннно, но валидно (исторически),
        // то *title - это все, до закрывающего разделителя*
        // Пример:
        // <{Я странный кат, у меня нет содержимого, зачем меня верстать?}>

        //  Находим конец строки
        let eolIndex = value.indexOf('\n', openingFollowingIndex);

        if (eolIndex < 0) {
            eolIndex = value.length;
        }

        const innerFirstIndex = findMatchBetween(defaultInlineChildren, value, openingFollowingIndex, eolIndex, i => {
            if (value[i] === '}' && value[i + 1] === '>') {
                return i;
            }

            return -1;
        });

        if (innerFirstIndex < 0) {
            // На той же строке не был найден закрывающий разделитель,
            // значит кат располагается на нескольких строках
            stateNode.innerFirstIndex = eolIndex;
        } else {
            // Кат располагается на одной строке
            stateNode.innerFirstIndex = innerFirstIndex;
        }

        stateNode.attributes.title = value.slice(openingFollowingIndex, stateNode.innerFirstIndex);
    },
    matchClosing: blockWomCut.matchClosing,
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

exports.blockWomCut = blockWomCut;
exports.inlineWomCut = inlineWomCut;
