const { expect } = require('chai');
const { parseTheContainerAt } = require('../parse-container-at');

const {
    blockWomCut,
    inlineWomCut,
} = require('../skip-blocks');

const createLA0ContainerParser = container =>
    (value, fromIndex) =>
        parseTheContainerAt(container, value, fromIndex);

const parseBlockWomCut = createLA0ContainerParser(blockWomCut);
const parseInlineWomCut = createLA0ContainerParser(inlineWomCut);

function traverse(node, fn) {
    fn(node);

    for (const childNode of node.children) {
        traverse(childNode, fn);
    }
}

function rmExtraProps(root) {
    traverse(root, node => {
        delete node.lineFeedCount;
    });
}

function runSamples(fn, samples) {
    for (const [args, expected] of samples) {
        it(JSON.stringify([...args]), () => {
            const node = fn(...args);

            rmExtraProps(node);

            expect(node).to.deep.equal(expected);
        });
    }
}

// 0 &&
describe('womCut(block)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['<', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['\n}>\n<{\n', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['\n}>', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<{\n', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<{\n1\n}>', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 5,
                    closingFollowingIndex: 7,
                    outerFirstIndex: 7,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 5,
                            outerFirstIndex: 5,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2\n}>\n3', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 5,
                    closingFollowingIndex: 7,
                    outerFirstIndex: 7,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 5,
                            outerFirstIndex: 5,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2\n<{\n3\n}>\n4\n}>\n5', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 5,
                            outerFirstIndex: 5,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 5,
                            openingFollowingIndex: 7,
                            innerFirstIndex: 7,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 7,
                                    openingFollowingIndex: 7,
                                    innerFirstIndex: 7,
                                    closingInitialIndex: 10,
                                    closingFollowingIndex: 10,
                                    outerFirstIndex: 10,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2x\n<{\n3\n}>\nx4\n}>\n6', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 17,
                    closingFollowingIndex: 19,
                    outerFirstIndex: 19,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 6,
                            outerFirstIndex: 6,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 6,
                            openingFollowingIndex: 8,
                            innerFirstIndex: 8,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 8,
                                    openingFollowingIndex: 8,
                                    innerFirstIndex: 8,
                                    closingInitialIndex: 11,
                                    closingFollowingIndex: 11,
                                    outerFirstIndex: 11,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 13,
                            openingFollowingIndex: 13,
                            innerFirstIndex: 13,
                            closingInitialIndex: 17,
                            closingFollowingIndex: 17,
                            outerFirstIndex: 17,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2\n<{\n3\n}>\n<{\n4\n}>\n5\n}>\n6', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 23,
                    closingFollowingIndex: 25,
                    outerFirstIndex: 25,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 5,
                            outerFirstIndex: 5,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 5,
                            openingFollowingIndex: 7,
                            innerFirstIndex: 7,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 7,
                                    openingFollowingIndex: 7,
                                    innerFirstIndex: 7,
                                    closingInitialIndex: 10,
                                    closingFollowingIndex: 10,
                                    outerFirstIndex: 10,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 13,
                            openingFollowingIndex: 15,
                            innerFirstIndex: 15,
                            closingInitialIndex: 18,
                            closingFollowingIndex: 20,
                            outerFirstIndex: 20,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 15,
                                    openingFollowingIndex: 15,
                                    innerFirstIndex: 15,
                                    closingInitialIndex: 18,
                                    closingFollowingIndex: 18,
                                    outerFirstIndex: 18,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 20,
                            openingFollowingIndex: 20,
                            innerFirstIndex: 20,
                            closingInitialIndex: 23,
                            closingFollowingIndex: 23,
                            outerFirstIndex: 23,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2\n<{\n3\n}>\nx\n<{\n4\n}>\n5\n}>\n6', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 25,
                    closingFollowingIndex: 27,
                    outerFirstIndex: 27,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 5,
                            outerFirstIndex: 5,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 5,
                            openingFollowingIndex: 7,
                            innerFirstIndex: 7,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 7,
                                    openingFollowingIndex: 7,
                                    innerFirstIndex: 7,
                                    closingInitialIndex: 10,
                                    closingFollowingIndex: 10,
                                    outerFirstIndex: 10,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 15,
                            openingFollowingIndex: 17,
                            innerFirstIndex: 17,
                            closingInitialIndex: 20,
                            closingFollowingIndex: 22,
                            outerFirstIndex: 22,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 17,
                                    openingFollowingIndex: 17,
                                    innerFirstIndex: 17,
                                    closingInitialIndex: 20,
                                    closingFollowingIndex: 20,
                                    outerFirstIndex: 20,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 22,
                            openingFollowingIndex: 22,
                            innerFirstIndex: 22,
                            closingInitialIndex: 25,
                            closingFollowingIndex: 25,
                            outerFirstIndex: 25,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2x\n<{\n3\n}>\n<{\n4\n}>\nx5\n}>\n6', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 25,
                    closingFollowingIndex: 27,
                    outerFirstIndex: 27,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 6,
                            outerFirstIndex: 6,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 6,
                            openingFollowingIndex: 8,
                            innerFirstIndex: 8,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 8,
                                    openingFollowingIndex: 8,
                                    innerFirstIndex: 8,
                                    closingInitialIndex: 11,
                                    closingFollowingIndex: 11,
                                    outerFirstIndex: 11,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 13,
                            openingFollowingIndex: 13,
                            innerFirstIndex: 13,
                            closingInitialIndex: 14,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 14,
                            openingFollowingIndex: 16,
                            innerFirstIndex: 16,
                            closingInitialIndex: 19,
                            closingFollowingIndex: 21,
                            outerFirstIndex: 21,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 16,
                                    openingFollowingIndex: 16,
                                    innerFirstIndex: 16,
                                    closingInitialIndex: 19,
                                    closingFollowingIndex: 19,
                                    outerFirstIndex: 19,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 21,
                            openingFollowingIndex: 21,
                            innerFirstIndex: 21,
                            closingInitialIndex: 25,
                            closingFollowingIndex: 25,
                            outerFirstIndex: 25,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2x\n<{\n3\n}>\nx\n<{\n4\n}>\nx5\n}>\n6', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 27,
                    closingFollowingIndex: 29,
                    outerFirstIndex: 29,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 6,
                            outerFirstIndex: 6,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 6,
                            openingFollowingIndex: 8,
                            innerFirstIndex: 8,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 8,
                                    openingFollowingIndex: 8,
                                    innerFirstIndex: 8,
                                    closingInitialIndex: 11,
                                    closingFollowingIndex: 11,
                                    outerFirstIndex: 11,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 13,
                            openingFollowingIndex: 13,
                            innerFirstIndex: 13,
                            closingInitialIndex: 16,
                            closingFollowingIndex: 16,
                            outerFirstIndex: 16,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 16,
                            openingFollowingIndex: 18,
                            innerFirstIndex: 18,
                            closingInitialIndex: 21,
                            closingFollowingIndex: 23,
                            outerFirstIndex: 23,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 18,
                                    openingFollowingIndex: 18,
                                    innerFirstIndex: 18,
                                    closingInitialIndex: 21,
                                    closingFollowingIndex: 21,
                                    outerFirstIndex: 21,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 23,
                            openingFollowingIndex: 23,
                            innerFirstIndex: 23,
                            closingInitialIndex: 27,
                            closingFollowingIndex: 27,
                            outerFirstIndex: 27,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n%%\nx\n%%\n}>\nb', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 11,
                    closingFollowingIndex: 13,
                    outerFirstIndex: 13,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 8,
                                    outerFirstIndex: 8,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n%%\nx\n%%\n\n%%\nx\n%%\n}>\nb', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 20,
                    closingFollowingIndex: 22,
                    outerFirstIndex: 22,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 8,
                                    outerFirstIndex: 8,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 12,
                            openingFollowingIndex: 14,
                            innerFirstIndex: 14,
                            closingInitialIndex: 17,
                            closingFollowingIndex: 19,
                            outerFirstIndex: 19,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 14,
                                    openingFollowingIndex: 14,
                                    innerFirstIndex: 14,
                                    closingInitialIndex: 17,
                                    closingFollowingIndex: 17,
                                    outerFirstIndex: 17,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 19,
                            openingFollowingIndex: 19,
                            innerFirstIndex: 19,
                            closingInitialIndex: 20,
                            closingFollowingIndex: 20,
                            outerFirstIndex: 20,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n%%\nx\n%%\n<{\n%%\nx\n%%\n}>\n%%\nx\n%%\n}>\nb', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 33,
                    closingFollowingIndex: 35,
                    outerFirstIndex: 35,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 8,
                                    outerFirstIndex: 8,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 11,
                            openingFollowingIndex: 13,
                            innerFirstIndex: 13,
                            closingInitialIndex: 22,
                            closingFollowingIndex: 24,
                            outerFirstIndex: 24,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 13,
                                    openingFollowingIndex: 13,
                                    innerFirstIndex: 13,
                                    closingInitialIndex: 14,
                                    closingFollowingIndex: 14,
                                    outerFirstIndex: 14,
                                    children: [],
                                },
                                {
                                    type: 'womFormatter',
                                    inline: false,
                                    attributes: {
                                        format: '',
                                        params: {},
                                    },
                                    openingInitialIndex: 14,
                                    openingFollowingIndex: 16,
                                    innerFirstIndex: 16,
                                    closingInitialIndex: 19,
                                    closingFollowingIndex: 21,
                                    outerFirstIndex: 21,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 16,
                                            openingFollowingIndex: 16,
                                            innerFirstIndex: 16,
                                            closingInitialIndex: 19,
                                            closingFollowingIndex: 19,
                                            outerFirstIndex: 19,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 21,
                                    openingFollowingIndex: 21,
                                    innerFirstIndex: 21,
                                    closingInitialIndex: 22,
                                    closingFollowingIndex: 22,
                                    outerFirstIndex: 22,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 24,
                            openingFollowingIndex: 24,
                            innerFirstIndex: 24,
                            closingInitialIndex: 25,
                            closingFollowingIndex: 25,
                            outerFirstIndex: 25,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 25,
                            openingFollowingIndex: 27,
                            innerFirstIndex: 27,
                            closingInitialIndex: 30,
                            closingFollowingIndex: 32,
                            outerFirstIndex: 32,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 27,
                                    openingFollowingIndex: 27,
                                    innerFirstIndex: 27,
                                    closingInitialIndex: 30,
                                    closingFollowingIndex: 30,
                                    outerFirstIndex: 30,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 32,
                            openingFollowingIndex: 32,
                            innerFirstIndex: 32,
                            closingInitialIndex: 33,
                            closingFollowingIndex: 33,
                            outerFirstIndex: 33,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2\n}>\n<{\n3\n}>\n4', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 5,
                    closingFollowingIndex: 7,
                    outerFirstIndex: 7,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 5,
                            outerFirstIndex: 5,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n2\n}>\n3\n<{\n4\n}>\n5', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 5,
                    closingFollowingIndex: 7,
                    outerFirstIndex: 7,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 5,
                            outerFirstIndex: 5,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n""\n}>\n""\n}>', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 14,
                    outerFirstIndex: 14,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womEscape',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 5,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: true,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 11,
                            openingFollowingIndex: 11,
                            innerFirstIndex: 11,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n%%%\n1\n%%%\n}>', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 13,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 6,
                                    innerFirstIndex: 6,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['<{\n%%%\n1\n%%%\n}>\n', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 13,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 6,
                                    innerFirstIndex: 6,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['<{\n%%%\n1\n%%%\n}>\n%%(md)\n%%%\n2\n%%%\n%%', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 13,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 3,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 6,
                                    innerFirstIndex: 6,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 9,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['<{title\ntest\n}>', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: 'title',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 7,
                    closingInitialIndex: 13,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 7,
                            openingFollowingIndex: 7,
                            innerFirstIndex: 7,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['<{title', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<{title\n', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {
                        title: 'title',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 7,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<{', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<{test}>\n<{\n}>', 0],
                {
                    type: 'womCut',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
        ],
    );

    runSamples(parseBlockWomCut, samples);
});

// 0 &&
describe('womCut(inline)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['<', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<{test}>\n<{\n}>', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: 'test',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 8,
                    outerFirstIndex: 8,
                    children: [],
                },
            ],
            [
                ['<{test}>', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: 'test',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 8,
                    outerFirstIndex: 8,
                    children: [],
                },
            ],
            [
                ['<{}>', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 2,
                    closingFollowingIndex: 4,
                    outerFirstIndex: 4,
                    children: [],
                },
            ],
            [
                ['<{<{}>}>', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: '<{}>',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 8,
                    outerFirstIndex: 8,
                    children: [],
                },
            ],
            [
                ['<{test\\}>}>', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: 'test\\}>',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 9,
                    closingInitialIndex: 9,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [],
                },
            ],
            [
                ['<{test}>\n', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: 'test',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 8,
                    outerFirstIndex: 8,
                    children: [],
                },
            ],
            [
                ['<{test\n}>', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: 'test',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 7,
                    closingFollowingIndex: 9,
                    outerFirstIndex: 9,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['<{title', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: 'title',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 7,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<{title\n', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: 'title',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 7,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['<{', 0],
                {
                    type: 'womCut',
                    inline: true,
                    attributes: {
                        title: '',
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
        ],
    );

    runSamples(parseInlineWomCut, samples);
});
