const { isEscaped } = require('../spec');
const {
    isLineOpening,
    skipLineFollowingSpacing,
    parseWomFormatterHeader,
    parseContainerParams,
} = require('../skip-blocks-utils');
const { createUnknownNode } = require('../parse-container-at');

// Wom форматтер кода (block)
const blockWomFormatter = {
    type: 'womFormatter',
    inline: false,
    possibleChildren: [],
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        if (value[openingInitialIndex] !== '%' || value[openingInitialIndex + 1] !== '%') {
            // Early return
            return;
        }

        // Невозможно заэкранировать открывающий разделитель блока,
        // потому что перед ним всегда есть \n, либо он в самом начале строки.
        // Поэтому на экранирование не проверяем.
        // Проверка начала строки - ниже
        let openingFollowingIndex = openingInitialIndex + 2;

        while (openingFollowingIndex < l && value[openingFollowingIndex] === '%') {
            openingFollowingIndex += 1;
        }

        let format = '';
        let params = '';
        let innerFirstIndex = openingFollowingIndex;

        if (value[openingFollowingIndex] === '(') {
            ({
                format,
                params,
                offset: innerFirstIndex,
            } = parseWomFormatterHeader(value, openingFollowingIndex));
        }

        if (!isLineOpening(value, openingInitialIndex) || skipLineFollowingSpacing(value, innerFirstIndex) < 0) {
            return;
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = innerFirstIndex;
        stateNode.attributes.format = format;
        stateNode.attributes.params = parseContainerParams(params);
    },
    matchClosing(value, closingInitialIndex_, stateNode) {
        let closingInitialIndex = closingInitialIndex_;
        const l = value.length;

        if (l - closingInitialIndex < 2) {
            // Закрывающий разделитель не можеть быть меньше 2
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Перед последовательностью разделителя
        // не может идти часть этой последовательности
        if (value[closingInitialIndex - 1] === '%') {
            return;
        }

        if (value[closingInitialIndex] !== '%' || value[closingInitialIndex + 1] !== '%') {
            return;
        }

        let closingFollowingIndex = closingInitialIndex + 2;

        // Собираем оставшуюся часть последовательности
        while (closingFollowingIndex < l && value[closingFollowingIndex] === '%') {
            closingFollowingIndex += 1;
        }

        const { openingInitialIndex, openingFollowingIndex } = stateNode;
        const openingDelimiterSize = openingFollowingIndex - openingInitialIndex;
        let closingDelimiterSize = closingFollowingIndex - closingInitialIndex;

        if (value[closingInitialIndex - 1] === '~') {
            if (closingDelimiterSize > openingDelimiterSize) {
                // Тильда может заэкранировать часть разделителя, только если
                // Закрывающий разделитель больше открывающего
                // как минимум на минимальную величину разделителя (2)
                // Делаем разделители равного размера
                closingInitialIndex += closingDelimiterSize - openingDelimiterSize;
                closingDelimiterSize = openingDelimiterSize;
            } else {
                // Если закрывающий разделитель меньше или равен открывающему
                // То ~ его экранирует
                return;
            }
        }

        // Но он может быть другого размера
        if (closingDelimiterSize > openingDelimiterSize) {
            // Если он больше, то отдает свою лишнюю левую часть внутрь форматтера
            closingInitialIndex += closingDelimiterSize - openingDelimiterSize;
        } else if (closingDelimiterSize < openingDelimiterSize) {
            // Если он меньше, то все сложне, тогда открывающий разделитель должен отдать
            // внутрь свою лишнюю правую часть
            stateNode.openingFollowingIndex -= openingDelimiterSize - closingDelimiterSize;

            // Надо подвинуть маркеры открывающего разделителя
            if (stateNode.attributes.format !== '') {
                // Если есть параметры, то их надо разобрать в текст
                stateNode.attributes.format = '';
                stateNode.attributes.params = {};
            }

            // Теперь надо подвинуть атрибуты контента
            // В контенте первой нодой может быть `unknown`, `womEscape` или пусто
            if (stateNode.children.length === 0) {
                // Пусто, надо запушить новую ноду в начало,
                // которая теперь будет там вместо параметро
                stateNode.children.push(createUnknownNode(
                    stateNode,
                    stateNode.openingFollowingIndex,
                    stateNode.innerFirstIndex,
                ));
            } else {
                const firstChild = stateNode.children[0];

                // Нераспознанный контент внутри, надо вытянуть его левый край
                firstChild.openingInitialIndex = stateNode.openingFollowingIndex;
                firstChild.openingFollowingIndex = stateNode.openingFollowingIndex;
                firstChild.innerFirstIndex = stateNode.openingFollowingIndex;
            }

            stateNode.innerFirstIndex = stateNode.openingFollowingIndex;
        }

        stateNode.closingInitialIndex = closingInitialIndex;
        stateNode.closingFollowingIndex = closingFollowingIndex;
        stateNode.outerFirstIndex = closingFollowingIndex;
    },
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

// Wom-форматтер кода (inline)
const inlineWomFormatter = {
    type: 'womFormatter',
    inline: true,
    possibleChildren: [],
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Открывающий разделитель может быть заэкранирован
        // Перед открывающим разделителем идет любой другой незаэкранированный символ
        if (isEscaped(value, openingInitialIndex) || (value[openingInitialIndex - 1] === '%' && !isEscaped(value, openingInitialIndex - 1))) {
            return;
        }

        if (value[openingInitialIndex] !== '%' || value[openingInitialIndex + 1] !== '%') {
            // Early return
            return;
        }

        let openingFollowingIndex = openingInitialIndex + 1;

        while (openingFollowingIndex < l && value[openingFollowingIndex] === '%') {
            openingFollowingIndex += 1;
        }

        let format = '';
        let params = '';
        let innerFirstIndex = openingFollowingIndex;

        if (value[openingFollowingIndex] === '(') {
            ({
                format,
                params,
                offset: innerFirstIndex,
            } = parseWomFormatterHeader(value, openingFollowingIndex));
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = innerFirstIndex;
        stateNode.attributes.format = format;
        stateNode.attributes.params = parseContainerParams(params);
    },
    matchClosing: blockWomFormatter.matchClosing,
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

exports.blockWomFormatter = blockWomFormatter;
exports.inlineWomFormatter = inlineWomFormatter;
