const { isEscaped } = require('../spec');
const {
    isLineOpening,
    skipLineFollowingSpacing,
    parseWomFormatterHeader,
    parseContainerParams,
} = require('../skip-blocks-utils');

const MD_FORMATS = Object.create(null);

MD_FORMATS.md = true;
MD_FORMATS.markdown = true;
MD_FORMATS.wacko = true;

const { defaultBlockChildren } = require('./possible-children');

// Wom обертка (block)
const blockWomMarkdown = {
    type: 'womMarkdown',
    inline: false,
    possibleChildren: defaultBlockChildren,
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        if (value[openingInitialIndex] !== '%' || value[openingInitialIndex + 1] !== '%') {
            // Early return
            return;
        }

        // Невозможно заэкранировать открывающий разделитель блока,
        // потому что перед ним всегда есть \n, либо он в самом начале строки.
        // Поэтому на экранирование не проверяем.
        // Проверка начала строки - ниже
        let openingFollowingIndex = openingInitialIndex + 2;

        while (openingFollowingIndex < l && value[openingFollowingIndex] === '%') {
            openingFollowingIndex += 1;
        }

        let format = '';
        let params = '';
        let innerFirstIndex = openingFollowingIndex;

        if (value[openingFollowingIndex] === '(') {
            ({
                format,
                params,
                offset: innerFirstIndex,
            } = parseWomFormatterHeader(value, openingFollowingIndex));
        }

        if (!MD_FORMATS[format]) {
            return;
        }

        if (!isLineOpening(value, openingInitialIndex)) {
            return;
        }

        // Если значение не в начале строки, проверяем многострочник это или нет
        // проверяем именно начальный индекс, для того, чтоб в списках, не делать блочный wom блок
        if (openingInitialIndex !== 0 && skipLineFollowingSpacing(value, innerFirstIndex) < 0) {
            return;
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = innerFirstIndex;
        stateNode.attributes.format = format;
        stateNode.attributes.params = parseContainerParams(params);
    },
    matchClosing(value, closingInitialIndex_, stateNode) {
        let closingInitialIndex = closingInitialIndex_;
        const l = value.length;

        if (l - closingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Закрывающий разделитель может быть заэкранирован
        // Перед закрывающим разделителем идет любой другой незаэкранированный символ
        if (isEscaped(value, closingInitialIndex) || (value[closingInitialIndex - 1] === '%' && !isEscaped(value, closingInitialIndex - 1))) {
            return;
        }

        if (value[closingInitialIndex] !== '%' || value[closingInitialIndex + 1] !== '%') {
            // Early return
            return;
        }

        let closingFollowingIndex = closingInitialIndex + 2;

        while (closingFollowingIndex < l && value[closingFollowingIndex] === '%') {
            closingFollowingIndex += 1;
        }

        const { openingInitialIndex, openingFollowingIndex } = stateNode;
        const openingDelimiterSize = openingFollowingIndex - openingInitialIndex;
        let closingDelimiterSize = closingFollowingIndex - closingInitialIndex;

        // Закрывающий разделитель womMarkdown не может быть короче открывающего
        if (closingDelimiterSize < openingDelimiterSize) {
            return;
        }

        // Закрывающий разделитель womMarkdown может быть длиннее чем открывающий
        if (closingDelimiterSize > openingDelimiterSize) {
            closingInitialIndex += closingDelimiterSize - openingDelimiterSize;
        }

        stateNode.closingInitialIndex = closingInitialIndex;
        stateNode.closingFollowingIndex = closingFollowingIndex;
        stateNode.outerFirstIndex = closingFollowingIndex;
    },
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

// Wom обертка (inline)
const inlineWomMarkdown = {
    type: 'womMarkdown',
    inline: true,
    possibleChildren: defaultBlockChildren,
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Открывающий разделитель может быть заэкранирован
        // Перед открывающим разделителем идет любой другой незаэкранированный символ
        if (isEscaped(value, openingInitialIndex) || (value[openingInitialIndex - 1] === '%' && !isEscaped(value, openingInitialIndex - 1))) {
            return;
        }

        if (value[openingInitialIndex] !== '%' || value[openingInitialIndex + 1] !== '%') {
            // Early return
            return;
        }

        let openingFollowingIndex = openingInitialIndex + 2;

        while (openingFollowingIndex < l && value[openingFollowingIndex] === '%') {
            openingFollowingIndex += 1;
        }

        let format = '';
        let params = '';
        let innerFirstIndex = openingFollowingIndex;

        if (value[openingFollowingIndex] === '(') {
            ({
                format,
                params,
                offset: innerFirstIndex,
            } = parseWomFormatterHeader(value, openingFollowingIndex));
        }

        if (!MD_FORMATS[format]) {
            return;
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = innerFirstIndex;
        stateNode.attributes.format = format;
        stateNode.attributes.params = parseContainerParams(params);
    },
    matchClosing: blockWomMarkdown.matchClosing,
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

exports.blockWomMarkdown = blockWomMarkdown;
exports.inlineWomMarkdown = inlineWomMarkdown;
