const { expect } = require('chai');
const { parseTheContainerAt } = require('../parse-container-at');

const {
    blockWomMarkdown,
    inlineWomMarkdown,
} = require('../skip-blocks');

const createLA0ContainerParser = container =>
    (value, fromIndex) =>
        parseTheContainerAt(container, value, fromIndex);

const parseBlockWomMarkdown = createLA0ContainerParser(blockWomMarkdown);
const parseInlineWomMarkdown = createLA0ContainerParser(inlineWomMarkdown);

function traverse(node, fn) {
    fn(node);

    for (const childNode of node.children) {
        traverse(childNode, fn);
    }
}

function rmExtraProps(root) {
    traverse(root, node => {
        delete node.lineFeedCount;
    });
}

function runSamples(fn, samples) {
    for (const [args, expected] of samples) {
        it(JSON.stringify([...args]), () => {
            const node = fn(...args);

            rmExtraProps(node);

            expect(node).to.deep.equal(expected);
        });
    }
}

// 0 &&
describe('womMarkdown(block)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['%%(md)\n~%%%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%x', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%%(md)\n1\\%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%%(md)\n\\%%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 9,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)1%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 7,
                    closingFollowingIndex: 9,
                    outerFirstIndex: 9,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n1\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 9,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n1\n%%\n2', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 9,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n2\n%%\n3', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 9,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%%(md)\na\n%%\nb\n%%\nc\n%%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 7,
                    closingInitialIndex: 20,
                    closingFollowingIndex: 23,
                    outerFirstIndex: 23,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 7,
                            innerFirstIndex: 7,
                            openingFollowingIndex: 7,
                            closingInitialIndex: 10,
                            outerFirstIndex: 10,
                            closingFollowingIndex: 10,
                            children: [],
                        },
                        {
                            type: 'womFormatter',
                            inline: false,
                            attributes: {
                                format: '',
                                params: {},
                            },
                            openingInitialIndex: 10,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 17,
                            outerFirstIndex: 17,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 12,
                                    innerFirstIndex: 12,
                                    openingFollowingIndex: 12,
                                    closingInitialIndex: 15,
                                    outerFirstIndex: 15,
                                    closingFollowingIndex: 15,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 17,
                            innerFirstIndex: 17,
                            openingFollowingIndex: 17,
                            closingInitialIndex: 20,
                            outerFirstIndex: 20,
                            closingFollowingIndex: 20,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\na\n%%%\nb\n%%%\nc\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 10,
                    closingFollowingIndex: 12,
                    outerFirstIndex: 12,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            innerFirstIndex: 6,
                            openingFollowingIndex: 6,
                            closingInitialIndex: 10,
                            outerFirstIndex: 10,
                            closingFollowingIndex: 10,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    innerFirstIndex: 9,
                                    openingFollowingIndex: 9,
                                    closingInitialIndex: 12,
                                    outerFirstIndex: 12,
                                    closingFollowingIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 15,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n%%\n<{x}>', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 12,
                                    closingFollowingIndex: 12,
                                    outerFirstIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 15,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\n2\n}>\n%%\n3', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    innerFirstIndex: 9,
                                    openingFollowingIndex: 9,
                                    closingInitialIndex: 12,
                                    outerFirstIndex: 12,
                                    closingFollowingIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 15,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n%%\nb', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    innerFirstIndex: 9,
                                    openingFollowingIndex: 9,
                                    closingInitialIndex: 12,
                                    outerFirstIndex: 12,
                                    closingFollowingIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 15,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n%%\n<{\nx\n}>\nb', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 12,
                                    closingFollowingIndex: 12,
                                    outerFirstIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 15,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n%%\n<{\nx\n}>\n%%(md)\n<{\nx\n}>\n%%\nb', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    innerFirstIndex: 9,
                                    openingFollowingIndex: 9,
                                    closingInitialIndex: 12,
                                    outerFirstIndex: 12,
                                    closingFollowingIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 15,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n<{\nx\n}>\n%%\n3', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 23,
                    closingFollowingIndex: 25,
                    outerFirstIndex: 25,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    innerFirstIndex: 9,
                                    openingFollowingIndex: 9,
                                    closingInitialIndex: 12,
                                    outerFirstIndex: 12,
                                    closingFollowingIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 15,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 15,
                            openingFollowingIndex: 17,
                            innerFirstIndex: 17,
                            closingInitialIndex: 20,
                            closingFollowingIndex: 22,
                            outerFirstIndex: 22,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 17,
                                    innerFirstIndex: 17,
                                    openingFollowingIndex: 17,
                                    closingInitialIndex: 20,
                                    outerFirstIndex: 20,
                                    closingFollowingIndex: 20,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 22,
                            innerFirstIndex: 22,
                            openingFollowingIndex: 22,
                            closingInitialIndex: 23,
                            outerFirstIndex: 23,
                            closingFollowingIndex: 23,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n%%%\nx\n%%%\n<{\nx\n}>\n%%\n3', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 16,
                    closingFollowingIndex: 18,
                    outerFirstIndex: 18,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    innerFirstIndex: 9,
                                    openingFollowingIndex: 9,
                                    closingInitialIndex: 12,
                                    outerFirstIndex: 12,
                                    closingFollowingIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 16,
                            outerFirstIndex: 16,
                            closingFollowingIndex: 16,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n%%%\n1\n%%%\n%%\n%%(md)\n%%%\n2\n%%%\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 10,
                    outerFirstIndex: 10,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['\n%%(md)\n<{\nx\n}>\n%%\ny\n<{\nx\n}>\ny\n%%\n<{\nx\n}>\n%%\nb', 1],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 1,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 7,
                    closingInitialIndex: 16,
                    closingFollowingIndex: 18,
                    outerFirstIndex: 18,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 7,
                            openingFollowingIndex: 7,
                            innerFirstIndex: 7,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 8,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 10,
                                    innerFirstIndex: 10,
                                    openingFollowingIndex: 10,
                                    closingInitialIndex: 13,
                                    outerFirstIndex: 13,
                                    closingFollowingIndex: 13,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 15,
                            innerFirstIndex: 15,
                            openingFollowingIndex: 15,
                            closingInitialIndex: 16,
                            outerFirstIndex: 16,
                            closingFollowingIndex: 16,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n<{\n%%(md)\n<{\ny\n}>\n%%\n}>\n<{\nx\n}>\n%%\n3', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 47,
                    closingFollowingIndex: 49,
                    outerFirstIndex: 49,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    innerFirstIndex: 9,
                                    openingFollowingIndex: 9,
                                    closingInitialIndex: 12,
                                    outerFirstIndex: 12,
                                    closingFollowingIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            innerFirstIndex: 14,
                            openingFollowingIndex: 14,
                            closingInitialIndex: 15,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 15,
                            openingFollowingIndex: 17,
                            innerFirstIndex: 17,
                            closingInitialIndex: 36,
                            closingFollowingIndex: 38,
                            outerFirstIndex: 38,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 17,
                                    openingFollowingIndex: 17,
                                    innerFirstIndex: 17,
                                    closingInitialIndex: 18,
                                    closingFollowingIndex: 18,
                                    outerFirstIndex: 18,
                                    children: [],
                                },
                                {
                                    type: 'womMarkdown',
                                    inline: false,
                                    attributes: {
                                        format: 'md',
                                        params: {},
                                    },
                                    openingInitialIndex: 18,
                                    openingFollowingIndex: 20,
                                    innerFirstIndex: 24,
                                    closingInitialIndex: 33,
                                    closingFollowingIndex: 35,
                                    outerFirstIndex: 35,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 24,
                                            openingFollowingIndex: 24,
                                            innerFirstIndex: 24,
                                            closingInitialIndex: 25,
                                            closingFollowingIndex: 25,
                                            outerFirstIndex: 25,
                                            children: [],
                                        },
                                        {
                                            inline: false,
                                            type: 'womCut',
                                            attributes: {
                                                title: '',
                                            },
                                            openingInitialIndex: 25,
                                            openingFollowingIndex: 27,
                                            innerFirstIndex: 27,
                                            closingInitialIndex: 30,
                                            closingFollowingIndex: 32,
                                            outerFirstIndex: 32,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: false,
                                                    attributes: {},
                                                    openingInitialIndex: 27,
                                                    innerFirstIndex: 27,
                                                    openingFollowingIndex: 27,
                                                    closingInitialIndex: 30,
                                                    outerFirstIndex: 30,
                                                    closingFollowingIndex: 30,
                                                    children: [],
                                                },
                                            ],
                                        },
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 32,
                                            innerFirstIndex: 32,
                                            openingFollowingIndex: 32,
                                            closingInitialIndex: 33,
                                            outerFirstIndex: 33,
                                            closingFollowingIndex: 33,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 35,
                                    innerFirstIndex: 35,
                                    openingFollowingIndex: 35,
                                    closingInitialIndex: 36,
                                    outerFirstIndex: 36,
                                    closingFollowingIndex: 36,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 38,
                            innerFirstIndex: 38,
                            openingFollowingIndex: 38,
                            closingInitialIndex: 39,
                            outerFirstIndex: 39,
                            closingFollowingIndex: 39,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 39,
                            openingFollowingIndex: 41,
                            innerFirstIndex: 41,
                            closingInitialIndex: 44,
                            closingFollowingIndex: 46,
                            outerFirstIndex: 46,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 41,
                                    innerFirstIndex: 41,
                                    openingFollowingIndex: 41,
                                    closingInitialIndex: 44,
                                    outerFirstIndex: 44,
                                    closingFollowingIndex: 44,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 46,
                            innerFirstIndex: 46,
                            openingFollowingIndex: 46,
                            closingInitialIndex: 47,
                            outerFirstIndex: 47,
                            closingFollowingIndex: 47,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n""\n%%\n""\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 16,
                    closingFollowingIndex: 18,
                    outerFirstIndex: 18,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womEscape',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 7,
                            innerFirstIndex: 9,
                            openingFollowingIndex: 9,
                            closingInitialIndex: 13,
                            outerFirstIndex: 15,
                            closingFollowingIndex: 15,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: true,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    innerFirstIndex: 9,
                                    openingFollowingIndex: 9,
                                    closingInitialIndex: 13,
                                    outerFirstIndex: 13,
                                    closingFollowingIndex: 13,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 15,
                            innerFirstIndex: 15,
                            openingFollowingIndex: 15,
                            closingInitialIndex: 16,
                            outerFirstIndex: 16,
                            closingFollowingIndex: 16,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md k="%%")\n1\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {
                            k: '%%',
                        },
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 13,
                    closingInitialIndex: 16,
                    closingFollowingIndex: 18,
                    outerFirstIndex: 18,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 13,
                            openingFollowingIndex: 13,
                            innerFirstIndex: 13,
                            closingInitialIndex: 16,
                            closingFollowingIndex: 16,
                            outerFirstIndex: 16,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md k="%%")\n1\n%% \n', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {
                            k: '%%',
                        },
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 13,
                    closingInitialIndex: 16,
                    closingFollowingIndex: 18,
                    outerFirstIndex: 18,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 13,
                            openingFollowingIndex: 13,
                            innerFirstIndex: 13,
                            closingInitialIndex: 16,
                            closingFollowingIndex: 16,
                            outerFirstIndex: 16,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md x y z)\n1\n%% \n', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {
                            x: null,
                            y: null,
                            z: null,
                        },
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 12,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\ntest\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 14,
                    outerFirstIndex: 14,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(x k="%%")\n1\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%%%(md)\n\n1\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 7,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%%%(md)\n1\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 7,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['%%(md)\n<{\n%%\nb\n%%\n}>\n%%\nb', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 21,
                    closingFollowingIndex: 23,
                    outerFirstIndex: 23,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 18,
                            closingFollowingIndex: 20,
                            outerFirstIndex: 20,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 10,
                                    closingFollowingIndex: 10,
                                    outerFirstIndex: 10,
                                    children: [],
                                },
                                {
                                    type: 'womFormatter',
                                    inline: false,
                                    attributes: {
                                        format: '',
                                        params: {},
                                    },
                                    openingInitialIndex: 10,
                                    openingFollowingIndex: 12,
                                    innerFirstIndex: 12,
                                    closingInitialIndex: 15,
                                    closingFollowingIndex: 17,
                                    outerFirstIndex: 17,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 12,
                                            innerFirstIndex: 12,
                                            openingFollowingIndex: 12,
                                            closingInitialIndex: 15,
                                            outerFirstIndex: 15,
                                            closingFollowingIndex: 15,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'unknown',

                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 17,
                                    openingFollowingIndex: 17,
                                    innerFirstIndex: 17,
                                    closingInitialIndex: 18,
                                    closingFollowingIndex: 18,
                                    outerFirstIndex: 18,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',

                            inline: false,
                            attributes: {},
                            openingInitialIndex: 20,
                            openingFollowingIndex: 20,
                            innerFirstIndex: 20,
                            closingInitialIndex: 21,
                            closingFollowingIndex: 21,
                            outerFirstIndex: 21,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n<{\n%%\ny\n%%\n}>\n%%\nb', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 29,
                    closingFollowingIndex: 31,
                    outerFirstIndex: 31,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 12,
                                    closingFollowingIndex: 12,
                                    outerFirstIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            openingFollowingIndex: 14,
                            innerFirstIndex: 14,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 15,
                            openingFollowingIndex: 17,
                            innerFirstIndex: 17,
                            closingInitialIndex: 26,
                            closingFollowingIndex: 28,
                            outerFirstIndex: 28,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 17,
                                    openingFollowingIndex: 17,
                                    innerFirstIndex: 17,
                                    closingInitialIndex: 18,
                                    closingFollowingIndex: 18,
                                    outerFirstIndex: 18,
                                    children: [],
                                },
                                {
                                    type: 'womFormatter',
                                    inline: false,
                                    attributes: {
                                        format: '',
                                        params: {},
                                    },
                                    openingInitialIndex: 18,
                                    openingFollowingIndex: 20,
                                    innerFirstIndex: 20,
                                    closingInitialIndex: 23,
                                    closingFollowingIndex: 25,
                                    outerFirstIndex: 25,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 20,
                                            openingFollowingIndex: 20,
                                            innerFirstIndex: 20,
                                            closingInitialIndex: 23,
                                            closingFollowingIndex: 23,
                                            outerFirstIndex: 23,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'unknown',

                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 25,
                                    openingFollowingIndex: 25,
                                    innerFirstIndex: 25,
                                    closingInitialIndex: 26,
                                    closingFollowingIndex: 26,
                                    outerFirstIndex: 26,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',

                            inline: false,
                            attributes: {},
                            openingInitialIndex: 28,
                            openingFollowingIndex: 28,
                            innerFirstIndex: 28,
                            closingInitialIndex: 29,
                            closingFollowingIndex: 29,
                            outerFirstIndex: 29,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)\n<{\nx\n}>\n<{\n%%\ny\n%%\n}>\n<{\nx\n}>\n%%\n3', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 37,
                    closingFollowingIndex: 39,
                    outerFirstIndex: 39,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 7,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 12,
                                    closingFollowingIndex: 12,
                                    outerFirstIndex: 12,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',

                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            openingFollowingIndex: 14,
                            innerFirstIndex: 14,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 15,
                            openingFollowingIndex: 17,
                            innerFirstIndex: 17,
                            closingInitialIndex: 26,
                            closingFollowingIndex: 28,
                            outerFirstIndex: 28,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 17,
                                    openingFollowingIndex: 17,
                                    innerFirstIndex: 17,
                                    closingInitialIndex: 18,
                                    closingFollowingIndex: 18,
                                    outerFirstIndex: 18,
                                    children: [],
                                },
                                {
                                    type: 'womFormatter',
                                    inline: false,
                                    attributes: {
                                        format: '',
                                        params: {},
                                    },
                                    openingInitialIndex: 18,
                                    openingFollowingIndex: 20,
                                    innerFirstIndex: 20,
                                    closingInitialIndex: 23,
                                    closingFollowingIndex: 25,
                                    outerFirstIndex: 25,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 20,
                                            openingFollowingIndex: 20,
                                            innerFirstIndex: 20,
                                            closingInitialIndex: 23,
                                            closingFollowingIndex: 23,
                                            outerFirstIndex: 23,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'unknown',

                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 25,
                                    openingFollowingIndex: 25,
                                    innerFirstIndex: 25,
                                    closingInitialIndex: 26,
                                    closingFollowingIndex: 26,
                                    outerFirstIndex: 26,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',

                            inline: false,
                            attributes: {},
                            openingInitialIndex: 28,
                            openingFollowingIndex: 28,
                            innerFirstIndex: 28,
                            closingInitialIndex: 29,
                            closingFollowingIndex: 29,
                            outerFirstIndex: 29,
                            children: [],
                        },
                        {
                            type: 'womCut',
                            inline: false,
                            attributes: {
                                title: '',
                            },
                            openingInitialIndex: 29,
                            openingFollowingIndex: 31,
                            innerFirstIndex: 31,
                            closingInitialIndex: 34,
                            closingFollowingIndex: 36,
                            outerFirstIndex: 36,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 31,
                                    openingFollowingIndex: 31,
                                    innerFirstIndex: 31,
                                    closingInitialIndex: 34,
                                    closingFollowingIndex: 34,
                                    outerFirstIndex: 34,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',

                            inline: false,
                            attributes: {},
                            openingInitialIndex: 36,
                            openingFollowingIndex: 36,
                            innerFirstIndex: 36,
                            closingInitialIndex: 37,
                            closingFollowingIndex: 37,
                            outerFirstIndex: 37,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['%%(md)\n<{\n%%\n', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 10,
                    closingFollowingIndex: 12,
                    outerFirstIndex: 12,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},

                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                [
                    '%%(markdown)\n' +
                    '   test\n' +
                    '%%',
                    0,
                ],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'markdown',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 12,
                    closingInitialIndex: 21,
                    closingFollowingIndex: 23,
                    outerFirstIndex: 23,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 21,
                            closingFollowingIndex: 21,
                            outerFirstIndex: 21,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['%%(wacko)\nx~%%\nx\n%%', 0],
                {
                    type: 'womMarkdown',
                    inline: false,
                    attributes: {
                        format: 'wacko',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 9,
                    closingInitialIndex: 17,
                    closingFollowingIndex: 19,
                    outerFirstIndex: 19,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 9,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [],
                        },
                        {
                            type: 'womEscape',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 11,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 14,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            children: [
                                {
                                    type: 'unknown',
                                    inline: true,
                                    attributes: {},
                                    openingInitialIndex: 12,
                                    openingFollowingIndex: 12,
                                    innerFirstIndex: 12,
                                    closingInitialIndex: 14,
                                    closingFollowingIndex: 14,
                                    outerFirstIndex: 14,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 14,
                            openingFollowingIndex: 14,
                            innerFirstIndex: 14,
                            closingInitialIndex: 17,
                            closingFollowingIndex: 17,
                            outerFirstIndex: 17,
                            children: [],
                        },
                    ],
                }
            ]
        ]
    );

    runSamples(parseBlockWomMarkdown, samples);
});

// 0 &&
describe('womMarkdown(inline)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['%%(md)<[]>test<[]>~%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%x', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%%%(md)\n1\\%%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 7,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%%(md)1\\%%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 9,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md)test%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 10,
                    closingFollowingIndex: 12,
                    outerFirstIndex: 12,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(md){[%%]}%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 10,
                    outerFirstIndex: 10,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['%%(x k="%%")1%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%%(x k="%%"){[%%]}%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['%%%(md)1%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 7,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['%%(md)1%%%2%%%3%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 10,
                    outerFirstIndex: 10,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 6,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 6,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                // empty
                ['%%(md)%%', 0],
                {
                    type: 'womMarkdown',
                    inline: true,
                    attributes: {
                        format: 'md',
                        params: {},
                    },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 6,
                    closingInitialIndex: 6,
                    closingFollowingIndex: 8,
                    outerFirstIndex: 8,
                    children: [],
                },
            ],
        ],
    );

    runSamples(parseInlineWomMarkdown, samples);
});
