const { isEscaped } = require('../spec');
const {
    isLineOpening,
    skipLineFollowingSpacing,
} = require('../skip-blocks-utils');

const { defaultBlockChildren } = require('./possible-children');

// internal womTableRow
const _blockWomTableRow = {
    type: 'womTableRow',
    inline: false,
    possibleChildren: [
        {
            type: 'womTableCell',
            inline: false,
            possibleChildren: defaultBlockChildren,
            matchOpening(value, openingInitialIndex, stateNode) {
                // Ячейка всегда начинается либо с уже сматченного открывающего разделителя строки,
                // либо с уже сматченного закрывающего разделителя предыдущей ячейки
                // Поэтому проверять открывающий разделитель ячейки не нужно
                stateNode.openingInitialIndex = openingInitialIndex;
                stateNode.openingFollowingIndex = openingInitialIndex + 1;
                stateNode.innerFirstIndex = openingInitialIndex + 1;
            },
            matchClosing(value, closingInitialIndex, stateNode) {
                // Закрывающий разделитель может быть заэкранирован
                if (value[closingInitialIndex] !== '|' || isEscaped(value, closingInitialIndex)) {
                    return;
                }

                stateNode.closingInitialIndex = closingInitialIndex;
                stateNode.closingFollowingIndex = closingInitialIndex + 1;
                // У закрывающих разделителей последней ячейки и строки есть общая часть
                stateNode.outerFirstIndex = closingInitialIndex;
            },
            enterSpacing() {},
            checkSpacing() {
                return true;
            },
        },
    ],
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Открывающий разделитель может быть заэкранирован
        if (!(value[openingInitialIndex] === '|' && value[openingInitialIndex + 1] === '|') || isEscaped(value, openingInitialIndex)) {
            return;
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingInitialIndex + 2;
        // У открывающих разделителей первой ячейки и строки есть общая часть
        stateNode.innerFirstIndex = openingInitialIndex + 1;
    },
    matchClosing(value, closingInitialIndex, stateNode) {
        if (!(stateNode.innerFirstIndex < closingInitialIndex)) {
            // Пустая таблица с одной ячейкой может выглядеть так:
            // #| |||| |#
            // Так как открытый ряд не перемещает указатель на всю свою длину,
            // то сразу же может сматчиться закрывающий разделитель.
            // Это неправильно
            return;
        }

        // Закрывающий разделитель может быть заэкранирован
        if (!(value[closingInitialIndex] === '|' && value[closingInitialIndex + 1] === '|') || isEscaped(value, closingInitialIndex)) {
            return;
        }

        stateNode.closingInitialIndex = closingInitialIndex;
        stateNode.closingFollowingIndex = closingInitialIndex + 2;
        stateNode.outerFirstIndex = closingInitialIndex + 2;
    },
    enterSpacing() {},
    /* istanbul ignore next */
    checkSpacing() {
        // Этот метод никогда не вызовется,
        // потому ячейки иидет вплотную к ряду
        return true;
    },
};

// Wom-таблица (block)
const blockWomTable = {
    type: 'womTable',
    inline: false,
    possibleChildren: [
        _blockWomTableRow,
    ],
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Невозможно заэкранировать открывающий разделитель блока,
        // потому что перед ним всегда есть \n, либо он в самом начале строки.
        // Поэтому на экранирование не проверяем.
        // Проверка начала строки - ниже
        if (!(value[openingInitialIndex] === '#' && value[openingInitialIndex + 1] === '|')) {
            return;
        }

        // Блочная таблица открывается либо через `#|`, либо через #||.
        // После разделителя должна заканчиваться строка,
        // поэтому тут нет магии про четное и нечетное количество палок.
        let openingFollowingIndex = openingInitialIndex + 2;

        if (!(l - openingInitialIndex < 3) && value[openingFollowingIndex] === '|') {
            openingFollowingIndex += 1;
            stateNode.attributes.kind = 'layout';
        }

        if (!isLineOpening(value, openingInitialIndex) || skipLineFollowingSpacing(value, openingFollowingIndex) < 0) {
            return;
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = openingFollowingIndex;
    },
    matchClosing(value, closingInitialIndex, stateNode) {
        const l = value.length;

        if (l - closingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Закрывающий разделитель может быть заэкранирован
        if (isEscaped(value, closingInitialIndex)) {
            return;
        }

        let closingFollowingIndex = closingInitialIndex + 2;

        // Любая таблица может закрываться как `||#` так и `|#`
        if (
            !(l - closingInitialIndex < 3) &&
            value[closingInitialIndex] === '|' &&
            value[closingInitialIndex + 1] === '|' &&
            value[closingInitialIndex + 2] === '#'
        ) {
            closingFollowingIndex += 1;
        } else if (!(value[closingInitialIndex] === '|' && value[closingInitialIndex + 1] === '#')) {
            return;
        }

        stateNode.closingInitialIndex = closingInitialIndex;
        stateNode.closingFollowingIndex = closingFollowingIndex;
        stateNode.outerFirstIndex = closingFollowingIndex;
    },
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

// Wom-таблица (inline)
const inlineWomTable = {
    type: 'womTable',
    inline: true,
    possibleChildren: [
        _blockWomTableRow,
    ],
    matchOpening(value, openingInitialIndex, stateNode) {
        const l = value.length;

        if (l - openingInitialIndex < 2) {
            // Bounds check, avoiding eager v8 deopt
            return;
        }

        // Открывающий разделитель может быть заэкранирован
        if (!(value[openingInitialIndex] === '#' && value[openingInitialIndex + 1] === '|') || isEscaped(value, openingInitialIndex)) {
            return;
        }

        // Таблица открывается либо через `#|`,
        // либо через `#||`. В строчной таблице сразу после открывающего разделителя
        // может открываться строка таблицы, например: `#|||`.
        // Но в этом случае надо открыть таблицу через `#|`.
        // То есть если после решетки идет четное количество труб - `#||`, нечетное - `#|`
        let openingFollowingIndex = openingInitialIndex + 2;

        while (openingFollowingIndex < l && value[openingFollowingIndex] === '|') { // 1 за решетку
            openingFollowingIndex += 1;
        }

        const pipeCount = openingFollowingIndex - openingInitialIndex - 1;

        openingFollowingIndex = openingInitialIndex + 2;

        if (pipeCount % 2 === 0) {
            // Четное количество
            openingFollowingIndex += 1;
            stateNode.attributes.kind = 'layout';
        }

        stateNode.openingInitialIndex = openingInitialIndex;
        stateNode.openingFollowingIndex = openingFollowingIndex;
        stateNode.innerFirstIndex = openingFollowingIndex;
    },
    matchClosing: blockWomTable.matchClosing,
    enterSpacing() {},
    checkSpacing() {
        return true;
    },
};

exports.blockWomTable = blockWomTable;
exports.inlineWomTable = inlineWomTable;
