const { expect } = require('chai');
const { parseTheContainerAt } = require('../parse-container-at');

const {
    blockWomTable,
    inlineWomTable,
} = require('../skip-blocks');

const createLA0ContainerParser = container =>
    (value, fromIndex) =>
        parseTheContainerAt(container, value, fromIndex);

const parseBlockWomTable = createLA0ContainerParser(blockWomTable);
const parseInlineWomTable = createLA0ContainerParser(inlineWomTable);

function traverse(node, fn) {
    fn(node);

    for (const childNode of node.children) {
        traverse(childNode, fn);
    }
}

function rmExtraProps(root) {
    traverse(root, node => {
        delete node.lineFeedCount;
    });
}

function runSamples(fn, samples) {
    for (const [args, expected] of samples) {
        it(JSON.stringify([...args]), () => {
            const node = fn(...args);

            rmExtraProps(node);

            expect(node).to.deep.equal(expected);
        });
    }
}

// 0 &&
describe('womTable(block)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['#|\n||x||\n||#', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 9,
                    closingFollowingIndex: 12,
                    outerFirstIndex: 12,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 7,
                                    outerFirstIndex: 6,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 8,
                            openingFollowingIndex: 8,
                            innerFirstIndex: 8,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [],
                        },
                    ],
                }
            ],
            [
                ['#', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['#|\n|', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['#|\n||', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['#|\n||x|', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['#|\n||1||\n|#', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 9,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 7,
                                    outerFirstIndex: 6,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 8,
                            openingFollowingIndex: 8,
                            innerFirstIndex: 8,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['#|\n||1||\\|#', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['#||\n||1||\n||#', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: { kind: 'layout' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 10,
                    closingFollowingIndex: 13,
                    outerFirstIndex: 13,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 3,
                            innerFirstIndex: 3,
                            closingInitialIndex: 4,
                            closingFollowingIndex: 4,
                            outerFirstIndex: 4,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 4,
                            openingFollowingIndex: 6,
                            innerFirstIndex: 5,
                            closingInitialIndex: 7,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 5,
                                    openingFollowingIndex: 6,
                                    innerFirstIndex: 6,
                                    closingInitialIndex: 7,
                                    closingFollowingIndex: 8,
                                    outerFirstIndex: 7,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 6,
                                            openingFollowingIndex: 6,
                                            innerFirstIndex: 6,
                                            closingInitialIndex: 7,
                                            closingFollowingIndex: 7,
                                            outerFirstIndex: 7,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 9,
                            openingFollowingIndex: 9,
                            innerFirstIndex: 9,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [],
                        },
                    ],
                },
            ],
            [
                [
                    '#|\n' +
                    '||\n%%\n' +
                    'code\n' +
                    '%%\n||\n' +
                    '|#',
                    0,
                ],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 20,
                    closingFollowingIndex: 22,
                    outerFirstIndex: 22,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 17,
                            closingFollowingIndex: 19,
                            outerFirstIndex: 19,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 17,
                                    closingFollowingIndex: 18,
                                    outerFirstIndex: 17,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                        {
                                            type: 'womFormatter',
                                            inline: false,
                                            attributes: {
                                                format: '',
                                                params: {},
                                            },
                                            openingInitialIndex: 6,
                                            openingFollowingIndex: 8,
                                            innerFirstIndex: 8,
                                            closingInitialIndex: 14,
                                            closingFollowingIndex: 16,
                                            outerFirstIndex: 16,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: false,
                                                    attributes: {},
                                                    openingInitialIndex: 8,
                                                    openingFollowingIndex: 8,
                                                    innerFirstIndex: 8,
                                                    closingInitialIndex: 14,
                                                    closingFollowingIndex: 14,
                                                    outerFirstIndex: 14,
                                                    children: [],
                                                },
                                            ],
                                        },
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 16,
                                            openingFollowingIndex: 16,
                                            innerFirstIndex: 16,
                                            closingInitialIndex: 17,
                                            closingFollowingIndex: 17,
                                            outerFirstIndex: 17,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 19,
                            openingFollowingIndex: 19,
                            innerFirstIndex: 19,
                            closingInitialIndex: 20,
                            closingFollowingIndex: 20,
                            outerFirstIndex: 20,
                            children: [],
                        },
                    ],
                },
            ],
            [
                [
                    '#|\n' +
                    '|| %%\n' +
                    'code\n' +
                    '%% ||\n' +
                    '|#',
                    0,
                ],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 20,
                    closingFollowingIndex: 22,
                    outerFirstIndex: 22,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 17,
                            closingFollowingIndex: 19,
                            outerFirstIndex: 19,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 17,
                                    closingFollowingIndex: 18,
                                    outerFirstIndex: 17,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                        {
                                            type: 'womFormatter',
                                            inline: true,
                                            attributes: {
                                                format: '',
                                                params: {},
                                            },
                                            openingInitialIndex: 6,
                                            openingFollowingIndex: 8,
                                            innerFirstIndex: 8,
                                            closingInitialIndex: 14,
                                            closingFollowingIndex: 16,
                                            outerFirstIndex: 16,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: true,
                                                    attributes: {},
                                                    openingInitialIndex: 8,
                                                    openingFollowingIndex: 8,
                                                    innerFirstIndex: 8,
                                                    closingInitialIndex: 14,
                                                    closingFollowingIndex: 14,
                                                    outerFirstIndex: 14,
                                                    children: [],
                                                },
                                            ],
                                        },
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 16,
                                            openingFollowingIndex: 16,
                                            innerFirstIndex: 16,
                                            closingInitialIndex: 17,
                                            closingFollowingIndex: 17,
                                            outerFirstIndex: 17,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 19,
                            openingFollowingIndex: 19,
                            innerFirstIndex: 19,
                            closingInitialIndex: 20,
                            closingFollowingIndex: 20,
                            outerFirstIndex: 20,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['#|\n||a| |# | b ||\n|| c | d ||\n|#', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 30,
                    closingFollowingIndex: 32,
                    outerFirstIndex: 32,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 17,
                            outerFirstIndex: 17,
                            children: [
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 7,
                                    outerFirstIndex: 6,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                    ],
                                },
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 7,
                                    innerFirstIndex: 7,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 8,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 7,
                                            openingFollowingIndex: 7,
                                            innerFirstIndex: 7,
                                            closingInitialIndex: 8,
                                            closingFollowingIndex: 8,
                                            outerFirstIndex: 8,
                                            children: [],
                                        },
                                    ],
                                },
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 8,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 11,
                                    closingFollowingIndex: 12,
                                    outerFirstIndex: 11,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 9,
                                            openingFollowingIndex: 9,
                                            innerFirstIndex: 9,
                                            closingInitialIndex: 11,
                                            closingFollowingIndex: 11,
                                            outerFirstIndex: 11,
                                            children: [],
                                        },
                                    ],
                                },
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 11,
                                    openingFollowingIndex: 12,
                                    innerFirstIndex: 12,
                                    closingInitialIndex: 15,
                                    closingFollowingIndex: 16,
                                    outerFirstIndex: 15,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 12,
                                            openingFollowingIndex: 12,
                                            innerFirstIndex: 12,
                                            closingInitialIndex: 15,
                                            closingFollowingIndex: 15,
                                            outerFirstIndex: 15,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 17,
                            openingFollowingIndex: 17,
                            innerFirstIndex: 17,
                            closingInitialIndex: 18,
                            closingFollowingIndex: 18,
                            outerFirstIndex: 18,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 18,
                            openingFollowingIndex: 20,
                            innerFirstIndex: 19,
                            closingInitialIndex: 27,
                            closingFollowingIndex: 29,
                            outerFirstIndex: 29,
                            children: [
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 19,
                                    openingFollowingIndex: 20,
                                    innerFirstIndex: 20,
                                    closingInitialIndex: 23,
                                    closingFollowingIndex: 24,
                                    outerFirstIndex: 23,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 20,
                                            openingFollowingIndex: 20,
                                            innerFirstIndex: 20,
                                            closingInitialIndex: 23,
                                            closingFollowingIndex: 23,
                                            outerFirstIndex: 23,
                                            children: [],
                                        },
                                    ],
                                },
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 23,
                                    openingFollowingIndex: 24,
                                    innerFirstIndex: 24,
                                    closingInitialIndex: 27,
                                    closingFollowingIndex: 28,
                                    outerFirstIndex: 27,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 24,
                                            openingFollowingIndex: 24,
                                            innerFirstIndex: 24,
                                            closingInitialIndex: 27,
                                            closingFollowingIndex: 27,
                                            outerFirstIndex: 27,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 29,
                            openingFollowingIndex: 29,
                            innerFirstIndex: 29,
                            closingInitialIndex: 30,
                            closingFollowingIndex: 30,
                            outerFirstIndex: 30,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['#|\n||\n#|\n||a| |# | b ||\n|| c | d ||\n|#', 6],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 6,
                    openingFollowingIndex: 8,
                    innerFirstIndex: 8,
                    closingInitialIndex: 36,
                    closingFollowingIndex: 38,
                    outerFirstIndex: 38,
                    children: [
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 8,
                            openingFollowingIndex: 8,
                            innerFirstIndex: 8,
                            closingInitialIndex: 9,
                            closingFollowingIndex: 9,
                            outerFirstIndex: 9,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 9,
                            openingFollowingIndex: 11,
                            innerFirstIndex: 10,
                            closingInitialIndex: 21,
                            closingFollowingIndex: 23,
                            outerFirstIndex: 23,
                            children: [
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 10,
                                    openingFollowingIndex: 11,
                                    innerFirstIndex: 11,
                                    closingInitialIndex: 12,
                                    closingFollowingIndex: 13,
                                    outerFirstIndex: 12,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 11,
                                            openingFollowingIndex: 11,
                                            innerFirstIndex: 11,
                                            closingInitialIndex: 12,
                                            closingFollowingIndex: 12,
                                            outerFirstIndex: 12,
                                            children: [],
                                        },
                                    ],
                                },
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 12,
                                    openingFollowingIndex: 13,
                                    innerFirstIndex: 13,
                                    closingInitialIndex: 14,
                                    closingFollowingIndex: 15,
                                    outerFirstIndex: 14,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 13,
                                            openingFollowingIndex: 13,
                                            innerFirstIndex: 13,
                                            closingInitialIndex: 14,
                                            closingFollowingIndex: 14,
                                            outerFirstIndex: 14,
                                            children: [],
                                        },
                                    ],
                                },
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 14,
                                    openingFollowingIndex: 15,
                                    innerFirstIndex: 15,
                                    closingInitialIndex: 17,
                                    closingFollowingIndex: 18,
                                    outerFirstIndex: 17,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 15,
                                            openingFollowingIndex: 15,
                                            innerFirstIndex: 15,
                                            closingInitialIndex: 17,
                                            closingFollowingIndex: 17,
                                            outerFirstIndex: 17,
                                            children: [],
                                        },
                                    ],
                                },
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 17,
                                    openingFollowingIndex: 18,
                                    innerFirstIndex: 18,
                                    closingInitialIndex: 21,
                                    closingFollowingIndex: 22,
                                    outerFirstIndex: 21,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 18,
                                            openingFollowingIndex: 18,
                                            innerFirstIndex: 18,
                                            closingInitialIndex: 21,
                                            closingFollowingIndex: 21,
                                            outerFirstIndex: 21,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 23,
                            openingFollowingIndex: 23,
                            innerFirstIndex: 23,
                            closingInitialIndex: 24,
                            closingFollowingIndex: 24,
                            outerFirstIndex: 24,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 24,
                            openingFollowingIndex: 26,
                            innerFirstIndex: 25,
                            closingInitialIndex: 33,
                            closingFollowingIndex: 35,
                            outerFirstIndex: 35,
                            children: [
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 25,
                                    openingFollowingIndex: 26,
                                    innerFirstIndex: 26,
                                    closingInitialIndex: 29,
                                    closingFollowingIndex: 30,
                                    outerFirstIndex: 29,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 26,
                                            openingFollowingIndex: 26,
                                            innerFirstIndex: 26,
                                            closingInitialIndex: 29,
                                            closingFollowingIndex: 29,
                                            outerFirstIndex: 29,
                                            children: [],
                                        },
                                    ],
                                },
                                {

                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 29,
                                    openingFollowingIndex: 30,
                                    innerFirstIndex: 30,
                                    closingInitialIndex: 33,
                                    closingFollowingIndex: 34,
                                    outerFirstIndex: 33,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 30,
                                            openingFollowingIndex: 30,
                                            innerFirstIndex: 30,
                                            closingInitialIndex: 33,
                                            closingFollowingIndex: 33,
                                            outerFirstIndex: 33,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 35,
                            openingFollowingIndex: 35,
                            innerFirstIndex: 35,
                            closingInitialIndex: 36,
                            closingFollowingIndex: 36,
                            outerFirstIndex: 36,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['#|\n||\n#|\n||a| |# | b ||\n|| c | d ||\n|#', 0],
                {
                    type: 'womTable',
                    inline: false,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
        ]
    );

    runSamples(parseBlockWomTable, samples);
});

// 0 &&
describe('womTable(inline)', () => {
    const samples = [];

    samples.push(
        ...[
            [
                ['#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: -1,
                    openingFollowingIndex: -1,
                    innerFirstIndex: -1,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['#||', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: { kind: 'layout' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['#|\n||1||\\|#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: -1,
                    closingFollowingIndex: -1,
                    outerFirstIndex: -1,
                    children: [],
                },
            ],
            [
                ['#||||%%|%%||||#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: { kind: 'layout' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'womTableRow',
                            inline: false,
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            attributes: {},
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 10,
                                    closingFollowingIndex: 11,
                                    outerFirstIndex: 10,
                                    children: [
                                        {
                                            type: 'womFormatter',
                                            inline: true,
                                            attributes: {
                                                format: '',
                                                params: {},
                                            },
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 7,
                                            innerFirstIndex: 7,
                                            closingInitialIndex: 8,
                                            closingFollowingIndex: 10,
                                            outerFirstIndex: 10,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: true,
                                                    attributes: {},
                                                    openingInitialIndex: 7,
                                                    openingFollowingIndex: 7,
                                                    innerFirstIndex: 7,
                                                    closingInitialIndex: 8,
                                                    closingFollowingIndex: 8,
                                                    outerFirstIndex: 8,
                                                    children: [],
                                                },
                                            ],
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                },
            ],
            [
                ['#| ||%%|%%|| |#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 13,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            attributes: {},
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 10,
                                    closingFollowingIndex: 11,
                                    outerFirstIndex: 10,
                                    children: [
                                        {
                                            type: 'womFormatter',
                                            inline: true,
                                            attributes: {
                                                format: '',
                                                params: {},
                                            },
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 7,
                                            innerFirstIndex: 7,
                                            closingInitialIndex: 8,
                                            closingFollowingIndex: 10,
                                            outerFirstIndex: 10,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: true,
                                                    attributes: {},
                                                    openingInitialIndex: 7,
                                                    openingFollowingIndex: 7,
                                                    innerFirstIndex: 7,
                                                    closingInitialIndex: 8,
                                                    closingFollowingIndex: 8,
                                                    outerFirstIndex: 8,
                                                    children: [],
                                                },
                                            ],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['#| ||a|b|| |#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 11,
                    closingFollowingIndex: 13,
                    outerFirstIndex: 13,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            attributes: {},
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 7,
                                    outerFirstIndex: 6,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 7,
                                    innerFirstIndex: 7,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 8,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 7,
                                            openingFollowingIndex: 7,
                                            innerFirstIndex: 7,
                                            closingInitialIndex: 8,
                                            closingFollowingIndex: 8,
                                            outerFirstIndex: 8,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 11,
                            closingFollowingIndex: 11,
                            outerFirstIndex: 11,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['#| || a | b || |#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 15,
                    closingFollowingIndex: 17,
                    outerFirstIndex: 17,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 14,
                            outerFirstIndex: 14,
                            attributes: {},
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 8,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 8,
                                            closingFollowingIndex: 8,
                                            outerFirstIndex: 8,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 8,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 12,
                                    closingFollowingIndex: 13,
                                    outerFirstIndex: 12,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 9,
                                            openingFollowingIndex: 9,
                                            innerFirstIndex: 9,
                                            closingInitialIndex: 12,
                                            closingFollowingIndex: 12,
                                            outerFirstIndex: 12,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 14,
                            openingFollowingIndex: 14,
                            innerFirstIndex: 14,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 15,
                            outerFirstIndex: 15,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['#| |||| |#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 10,
                    outerFirstIndex: 10,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 5,
                                    closingFollowingIndex: 6,
                                    outerFirstIndex: 5,
                                    children: [],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 7,
                            openingFollowingIndex: 7,
                            innerFirstIndex: 7,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                // NOTE: После 2 сначала закрывается ряд,
                // а третья труба игнорится, как контент между строками таблицы
                // Такое же поведение было в старом форматере
                ['#| ||1|2||| |#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 12,
                    closingFollowingIndex: 14,
                    outerFirstIndex: 14,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 8,
                            closingFollowingIndex: 10,
                            outerFirstIndex: 10,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 7,
                                    outerFirstIndex: 6,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 7,
                                    innerFirstIndex: 7,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 8,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 7,
                                            openingFollowingIndex: 7,
                                            innerFirstIndex: 7,
                                            closingInitialIndex: 8,
                                            closingFollowingIndex: 8,
                                            outerFirstIndex: 8,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 10,
                            openingFollowingIndex: 10,
                            innerFirstIndex: 10,
                            closingInitialIndex: 12,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [],
                        },
                    ],
                },
            ],
            [
                ['#| ||1|2| || |#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 13,
                    closingFollowingIndex: 15,
                    outerFirstIndex: 15,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 10,
                            closingFollowingIndex: 12,
                            outerFirstIndex: 12,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 7,
                                    outerFirstIndex: 6,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 6,
                                    openingFollowingIndex: 7,
                                    innerFirstIndex: 7,
                                    closingInitialIndex: 8,
                                    closingFollowingIndex: 9,
                                    outerFirstIndex: 8,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 7,
                                            openingFollowingIndex: 7,
                                            innerFirstIndex: 7,
                                            closingInitialIndex: 8,
                                            closingFollowingIndex: 8,
                                            outerFirstIndex: 8,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 8,
                                    openingFollowingIndex: 9,
                                    innerFirstIndex: 9,
                                    closingInitialIndex: 10,
                                    closingFollowingIndex: 11,
                                    outerFirstIndex: 10,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 9,
                                            openingFollowingIndex: 9,
                                            innerFirstIndex: 9,
                                            closingInitialIndex: 10,
                                            closingFollowingIndex: 10,
                                            outerFirstIndex: 10,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 12,
                            openingFollowingIndex: 12,
                            innerFirstIndex: 12,
                            closingInitialIndex: 13,
                            closingFollowingIndex: 13,
                            outerFirstIndex: 13,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                // Не конфликтуем с womMonospace
                ['#||||test|##test##|test||||#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: { kind: 'layout' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 25,
                    closingFollowingIndex: 28,
                    outerFirstIndex: 28,
                    children: [
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 23,
                            closingFollowingIndex: 25,
                            outerFirstIndex: 25,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 10,
                                    outerFirstIndex: 9,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 9,
                                            closingFollowingIndex: 9,
                                            outerFirstIndex: 9,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    openingFollowingIndex: 10,
                                    innerFirstIndex: 10,
                                    closingInitialIndex: 18,
                                    closingFollowingIndex: 19,
                                    outerFirstIndex: 18,
                                    children: [
                                        {
                                            type: 'womMonospace',
                                            inline: true,
                                            attributes: {},
                                            openingInitialIndex: 10,
                                            openingFollowingIndex: 12,
                                            innerFirstIndex: 12,
                                            closingInitialIndex: 16,
                                            closingFollowingIndex: 18,
                                            outerFirstIndex: 18,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: true,
                                                    attributes: {},
                                                    openingInitialIndex: 12,
                                                    openingFollowingIndex: 12,
                                                    innerFirstIndex: 12,
                                                    closingInitialIndex: 16,
                                                    closingFollowingIndex: 16,
                                                    outerFirstIndex: 16,
                                                    children: [],
                                                },
                                            ],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 18,
                                    openingFollowingIndex: 19,
                                    innerFirstIndex: 19,
                                    closingInitialIndex: 23,
                                    closingFollowingIndex: 24,
                                    outerFirstIndex: 23,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 19,
                                            openingFollowingIndex: 19,
                                            innerFirstIndex: 19,
                                            closingInitialIndex: 23,
                                            closingFollowingIndex: 23,
                                            outerFirstIndex: 23,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                },
            ],
            [
                // Разделитель внутри womMonospace
                ['#||||test|##t|st##|test||||#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: { kind: 'layout' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 25,
                    closingFollowingIndex: 28,
                    outerFirstIndex: 28,
                    children: [
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 23,
                            closingFollowingIndex: 25,
                            outerFirstIndex: 25,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 9,
                                    closingFollowingIndex: 10,
                                    outerFirstIndex: 9,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 9,
                                            closingFollowingIndex: 9,
                                            outerFirstIndex: 9,
                                            children: [],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 9,
                                    openingFollowingIndex: 10,
                                    innerFirstIndex: 10,
                                    closingInitialIndex: 18,
                                    closingFollowingIndex: 19,
                                    outerFirstIndex: 18,
                                    children: [
                                        {
                                            type: 'womMonospace',
                                            inline: true,
                                            attributes: {},
                                            openingInitialIndex: 10,
                                            openingFollowingIndex: 12,
                                            innerFirstIndex: 12,
                                            closingInitialIndex: 16,
                                            closingFollowingIndex: 18,
                                            outerFirstIndex: 18,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: true,
                                                    attributes: {},
                                                    openingInitialIndex: 12,
                                                    openingFollowingIndex: 12,
                                                    innerFirstIndex: 12,
                                                    closingInitialIndex: 16,
                                                    closingFollowingIndex: 16,
                                                    outerFirstIndex: 16,
                                                    children: [],
                                                },
                                            ],
                                        },
                                    ],
                                },
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 18,
                                    openingFollowingIndex: 19,
                                    innerFirstIndex: 19,
                                    closingInitialIndex: 23,
                                    closingFollowingIndex: 24,
                                    outerFirstIndex: 23,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 19,
                                            openingFollowingIndex: 19,
                                            innerFirstIndex: 19,
                                            closingInitialIndex: 23,
                                            closingFollowingIndex: 23,
                                            outerFirstIndex: 23,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['#| || ~|a||s||d || |#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 19,
                    closingFollowingIndex: 21,
                    outerFirstIndex: 21,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 16,
                            closingFollowingIndex: 18,
                            outerFirstIndex: 18,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 16,
                                    closingFollowingIndex: 17,
                                    outerFirstIndex: 16,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                        {
                                            type: 'womEscape',
                                            inline: true,
                                            attributes: {},
                                            openingInitialIndex: 6,
                                            openingFollowingIndex: 7,
                                            innerFirstIndex: 7,
                                            closingInitialIndex: 15,
                                            closingFollowingIndex: 15,
                                            outerFirstIndex: 15,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: true,
                                                    attributes: {},
                                                    openingInitialIndex: 7,
                                                    openingFollowingIndex: 7,
                                                    innerFirstIndex: 7,
                                                    closingInitialIndex: 15,
                                                    closingFollowingIndex: 15,
                                                    outerFirstIndex: 15,
                                                    children: [],
                                                },
                                            ],
                                        },
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 15,
                                            openingFollowingIndex: 15,
                                            innerFirstIndex: 15,
                                            closingInitialIndex: 16,
                                            closingFollowingIndex: 16,
                                            outerFirstIndex: 16,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 18,
                            openingFollowingIndex: 18,
                            innerFirstIndex: 18,
                            closingInitialIndex: 19,
                            closingFollowingIndex: 19,
                            outerFirstIndex: 19,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['#|\n||foo ~| bar||\n|#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 18,
                    closingFollowingIndex: 20,
                    outerFirstIndex: 20,
                    children: [
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 2,
                            innerFirstIndex: 2,
                            closingInitialIndex: 3,
                            closingFollowingIndex: 3,
                            outerFirstIndex: 3,
                            children: [],
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 15,
                            closingFollowingIndex: 17,
                            outerFirstIndex: 17,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 15,
                                    closingFollowingIndex: 16,
                                    outerFirstIndex: 15,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 9,
                                            closingFollowingIndex: 9,
                                            outerFirstIndex: 9,
                                            children: [],
                                        },
                                        {
                                            type: 'womEscape',
                                            inline: true,
                                            attributes: {},
                                            openingInitialIndex: 9,
                                            openingFollowingIndex: 10,
                                            innerFirstIndex: 10,
                                            closingInitialIndex: 11,
                                            closingFollowingIndex: 11,
                                            outerFirstIndex: 11,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: true,
                                                    attributes: {},
                                                    openingInitialIndex: 10,
                                                    openingFollowingIndex: 10,
                                                    innerFirstIndex: 10,
                                                    closingInitialIndex: 11,
                                                    closingFollowingIndex: 11,
                                                    outerFirstIndex: 11,
                                                    children: [],
                                                },
                                            ],
                                        },
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 11,
                                            openingFollowingIndex: 11,
                                            innerFirstIndex: 11,
                                            closingInitialIndex: 15,
                                            closingFollowingIndex: 15,
                                            outerFirstIndex: 15,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            type: 'unknown',
                            inline: true,
                            attributes: {},
                            openingInitialIndex: 17,
                            openingFollowingIndex: 17,
                            innerFirstIndex: 17,
                            closingInitialIndex: 18,
                            closingFollowingIndex: 18,
                            outerFirstIndex: 18,
                            children: [],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['#|||a|||#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: {},
                    openingInitialIndex: 0,
                    openingFollowingIndex: 2,
                    innerFirstIndex: 2,
                    closingInitialIndex: 7,
                    closingFollowingIndex: 9,
                    outerFirstIndex: 9,
                    children: [
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 2,
                            openingFollowingIndex: 4,
                            innerFirstIndex: 3,
                            closingInitialIndex: 5,
                            closingFollowingIndex: 7,
                            outerFirstIndex: 7,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 3,
                                    openingFollowingIndex: 4,
                                    innerFirstIndex: 4,
                                    closingInitialIndex: 5,
                                    closingFollowingIndex: 6,
                                    outerFirstIndex: 5,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 4,
                                            openingFollowingIndex: 4,
                                            innerFirstIndex: 4,
                                            closingInitialIndex: 5,
                                            closingFollowingIndex: 5,
                                            outerFirstIndex: 5,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                },
            ],
            [
                ['#||||a|||#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: { kind: 'layout' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 10,
                    outerFirstIndex: 10,
                    children: [
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 7,
                                    outerFirstIndex: 6,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                },
            ],
            [
                ['#||||a||||#', 0],
                {
                    type: 'womTable',
                    inline: true,
                    attributes: { kind: 'layout' },
                    openingInitialIndex: 0,
                    openingFollowingIndex: 3,
                    innerFirstIndex: 3,
                    closingInitialIndex: 8,
                    closingFollowingIndex: 11,
                    outerFirstIndex: 11,
                    children: [
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            openingInitialIndex: 3,
                            openingFollowingIndex: 5,
                            innerFirstIndex: 4,
                            closingInitialIndex: 6,
                            closingFollowingIndex: 8,
                            outerFirstIndex: 8,
                            children: [
                                {
                                    type: 'womTableCell',
                                    inline: false,
                                    attributes: {},
                                    openingInitialIndex: 4,
                                    openingFollowingIndex: 5,
                                    innerFirstIndex: 5,
                                    closingInitialIndex: 6,
                                    closingFollowingIndex: 7,
                                    outerFirstIndex: 6,
                                    children: [
                                        {
                                            type: 'unknown',
                                            inline: false,
                                            attributes: {},
                                            openingInitialIndex: 5,
                                            openingFollowingIndex: 5,
                                            innerFirstIndex: 5,
                                            closingInitialIndex: 6,
                                            closingFollowingIndex: 6,
                                            outerFirstIndex: 6,
                                            children: [],
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                },
            ],
        ],
    );

    samples.push(
        ...[
            [
                ['#||\n|| cell10 | #| ||cell11.0|| |# | #| ||cell12.0|| |# | cell13 ||\n|| cell20 | #| ||cell21.0|| |# | #| ||cell22.0|| |# | cell23 ||\n||#', 0],
                {
                    attributes: {
                        kind: 'layout',
                    },
                    children: [
                        {
                            attributes: {},
                            children: [],
                            closingFollowingIndex: 4,
                            closingInitialIndex: 4,
                            inline: true,
                            innerFirstIndex: 3,
                            openingFollowingIndex: 3,
                            openingInitialIndex: 3,
                            outerFirstIndex: 4,
                            type: 'unknown',
                        },
                        {
                            type: 'womTableRow',
                            inline: false,
                            attributes: {},
                            closingFollowingIndex: 67,
                            closingInitialIndex: 65,
                            innerFirstIndex: 5,
                            openingFollowingIndex: 6,
                            openingInitialIndex: 4,
                            outerFirstIndex: 67,
                            children: [
                                {
                                    attributes: {},
                                    children: [
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 14,
                                            closingInitialIndex: 14,
                                            inline: false,
                                            innerFirstIndex: 6,
                                            openingFollowingIndex: 6,
                                            openingInitialIndex: 6,
                                            outerFirstIndex: 14,
                                            type: 'unknown',
                                        },
                                    ],
                                    closingFollowingIndex: 15,
                                    closingInitialIndex: 14,
                                    inline: false,
                                    innerFirstIndex: 6,
                                    openingFollowingIndex: 6,
                                    openingInitialIndex: 5,
                                    outerFirstIndex: 14,
                                    type: 'womTableCell',
                                },
                                {
                                    attributes: {},
                                    children: [
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 16,
                                            closingInitialIndex: 16,
                                            inline: false,
                                            innerFirstIndex: 15,
                                            openingFollowingIndex: 15,
                                            openingInitialIndex: 15,
                                            outerFirstIndex: 16,
                                            type: 'unknown',
                                        },
                                        {
                                            attributes: {},
                                            children: [
                                                {
                                                    attributes: {},
                                                    children: [],
                                                    closingFollowingIndex: 19,
                                                    closingInitialIndex: 19,
                                                    inline: true,
                                                    innerFirstIndex: 18,
                                                    openingFollowingIndex: 18,
                                                    openingInitialIndex: 18,
                                                    outerFirstIndex: 19,
                                                    type: 'unknown',
                                                },
                                                {
                                                    attributes: {},
                                                    children: [
                                                        {
                                                            attributes: {},
                                                            children: [
                                                                {
                                                                    attributes: {},
                                                                    children: [],
                                                                    closingFollowingIndex: 29,
                                                                    closingInitialIndex: 29,
                                                                    inline: false,
                                                                    innerFirstIndex: 21,
                                                                    openingFollowingIndex: 21,
                                                                    openingInitialIndex: 21,
                                                                    outerFirstIndex: 29,
                                                                    type: 'unknown',
                                                                },
                                                            ],
                                                            closingFollowingIndex: 30,
                                                            closingInitialIndex: 29,
                                                            inline: false,
                                                            innerFirstIndex: 21,
                                                            openingFollowingIndex: 21,
                                                            openingInitialIndex: 20,
                                                            outerFirstIndex: 29,
                                                            type: 'womTableCell',
                                                        },
                                                    ],
                                                    closingFollowingIndex: 31,
                                                    closingInitialIndex: 29,
                                                    inline: false,
                                                    innerFirstIndex: 20,
                                                    openingFollowingIndex: 21,
                                                    openingInitialIndex: 19,
                                                    outerFirstIndex: 31,
                                                    type: 'womTableRow',
                                                },
                                                {
                                                    attributes: {},
                                                    children: [],
                                                    closingFollowingIndex: 32,
                                                    closingInitialIndex: 32,
                                                    inline: true,
                                                    innerFirstIndex: 31,
                                                    openingFollowingIndex: 31,
                                                    openingInitialIndex: 31,
                                                    outerFirstIndex: 32,
                                                    type: 'unknown',
                                                },
                                            ],
                                            closingFollowingIndex: 34,
                                            closingInitialIndex: 32,
                                            inline: true,
                                            innerFirstIndex: 18,
                                            openingFollowingIndex: 18,
                                            openingInitialIndex: 16,
                                            outerFirstIndex: 34,
                                            type: 'womTable',
                                        },
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 35,
                                            closingInitialIndex: 35,
                                            inline: false,
                                            innerFirstIndex: 34,
                                            openingFollowingIndex: 34,
                                            openingInitialIndex: 34,
                                            outerFirstIndex: 35,
                                            type: 'unknown',
                                        },
                                    ],
                                    closingFollowingIndex: 36,
                                    closingInitialIndex: 35,
                                    inline: false,
                                    innerFirstIndex: 15,
                                    openingFollowingIndex: 15,
                                    openingInitialIndex: 14,
                                    outerFirstIndex: 35,
                                    type: 'womTableCell',
                                },
                                {
                                    attributes: {},
                                    children: [
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 37,
                                            closingInitialIndex: 37,
                                            inline: false,
                                            innerFirstIndex: 36,
                                            openingFollowingIndex: 36,
                                            openingInitialIndex: 36,
                                            outerFirstIndex: 37,
                                            type: 'unknown',
                                        },
                                        {
                                            type: 'womTable',
                                            inline: true,
                                            attributes: {},
                                            openingInitialIndex: 37,
                                            openingFollowingIndex: 39,
                                            innerFirstIndex: 39,
                                            closingInitialIndex: 53,
                                            closingFollowingIndex: 55,
                                            outerFirstIndex: 55,
                                            children: [
                                                {
                                                    attributes: {},
                                                    children: [],
                                                    inline: true,
                                                    openingInitialIndex: 39,
                                                    openingFollowingIndex: 39,
                                                    innerFirstIndex: 39,
                                                    closingInitialIndex: 40,
                                                    closingFollowingIndex: 40,
                                                    outerFirstIndex: 40,
                                                    type: 'unknown',
                                                },
                                                {
                                                    type: 'womTableRow',
                                                    inline: false,
                                                    attributes: {},
                                                    closingFollowingIndex: 52,
                                                    closingInitialIndex: 50,
                                                    innerFirstIndex: 41,
                                                    openingFollowingIndex: 42,
                                                    openingInitialIndex: 40,
                                                    outerFirstIndex: 52,
                                                    children: [
                                                        {
                                                            attributes: {},
                                                            children: [
                                                                {
                                                                    attributes: {},
                                                                    children: [],
                                                                    closingFollowingIndex: 50,
                                                                    closingInitialIndex: 50,
                                                                    inline: false,
                                                                    innerFirstIndex: 42,
                                                                    openingFollowingIndex: 42,
                                                                    openingInitialIndex: 42,
                                                                    outerFirstIndex: 50,
                                                                    type: 'unknown',
                                                                },
                                                            ],
                                                            closingFollowingIndex: 51,
                                                            closingInitialIndex: 50,
                                                            inline: false,
                                                            innerFirstIndex: 42,
                                                            openingFollowingIndex: 42,
                                                            openingInitialIndex: 41,
                                                            outerFirstIndex: 50,
                                                            type: 'womTableCell',
                                                        },
                                                    ],
                                                },
                                                {
                                                    attributes: {},
                                                    children: [],
                                                    closingFollowingIndex: 53,
                                                    closingInitialIndex: 53,
                                                    inline: true,
                                                    innerFirstIndex: 52,
                                                    openingFollowingIndex: 52,
                                                    openingInitialIndex: 52,
                                                    outerFirstIndex: 53,
                                                    type: 'unknown',
                                                },
                                            ],
                                        },
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 56,
                                            closingInitialIndex: 56,
                                            inline: false,
                                            innerFirstIndex: 55,
                                            openingFollowingIndex: 55,
                                            openingInitialIndex: 55,
                                            outerFirstIndex: 56,
                                            type: 'unknown',
                                        },
                                    ],
                                    closingFollowingIndex: 57,
                                    closingInitialIndex: 56,
                                    inline: false,
                                    innerFirstIndex: 36,
                                    openingFollowingIndex: 36,
                                    openingInitialIndex: 35,
                                    outerFirstIndex: 56,
                                    type: 'womTableCell',
                                },
                                {
                                    attributes: {},
                                    children: [
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 65,
                                            closingInitialIndex: 65,
                                            inline: false,
                                            innerFirstIndex: 57,
                                            openingFollowingIndex: 57,
                                            openingInitialIndex: 57,
                                            outerFirstIndex: 65,
                                            type: 'unknown',
                                        },
                                    ],
                                    closingFollowingIndex: 66,
                                    closingInitialIndex: 65,
                                    inline: false,
                                    innerFirstIndex: 57,
                                    openingFollowingIndex: 57,
                                    openingInitialIndex: 56,
                                    outerFirstIndex: 65,
                                    type: 'womTableCell',
                                },
                            ],
                        },
                        {
                            attributes: {},
                            children: [],
                            closingFollowingIndex: 68,
                            closingInitialIndex: 68,
                            inline: true,
                            innerFirstIndex: 67,
                            openingFollowingIndex: 67,
                            openingInitialIndex: 67,
                            outerFirstIndex: 68,
                            type: 'unknown',
                        },
                        {
                            attributes: {},
                            children: [
                                {
                                    attributes: {},
                                    children: [
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 78,
                                            closingInitialIndex: 78,
                                            inline: false,
                                            innerFirstIndex: 70,
                                            openingFollowingIndex: 70,
                                            openingInitialIndex: 70,
                                            outerFirstIndex: 78,
                                            type: 'unknown',
                                        },
                                    ],
                                    closingFollowingIndex: 79,
                                    closingInitialIndex: 78,
                                    inline: false,
                                    innerFirstIndex: 70,
                                    openingFollowingIndex: 70,
                                    openingInitialIndex: 69,
                                    outerFirstIndex: 78,
                                    type: 'womTableCell',
                                },
                                {
                                    attributes: {},
                                    children: [
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 80,
                                            closingInitialIndex: 80,
                                            inline: false,
                                            innerFirstIndex: 79,
                                            openingFollowingIndex: 79,
                                            openingInitialIndex: 79,
                                            outerFirstIndex: 80,
                                            type: 'unknown',
                                        },
                                        {
                                            attributes: {},
                                            children: [
                                                {
                                                    attributes: {},
                                                    children: [],
                                                    closingFollowingIndex: 83,
                                                    closingInitialIndex: 83,
                                                    inline: true,
                                                    innerFirstIndex: 82,
                                                    openingFollowingIndex: 82,
                                                    openingInitialIndex: 82,
                                                    outerFirstIndex: 83,
                                                    type: 'unknown',
                                                },
                                                {
                                                    attributes: {},
                                                    children: [
                                                        {
                                                            attributes: {},
                                                            children: [
                                                                {
                                                                    attributes: {},
                                                                    children: [],
                                                                    closingFollowingIndex: 93,
                                                                    closingInitialIndex: 93,
                                                                    inline: false,
                                                                    innerFirstIndex: 85,
                                                                    openingFollowingIndex: 85,
                                                                    openingInitialIndex: 85,
                                                                    outerFirstIndex: 93,
                                                                    type: 'unknown',
                                                                },
                                                            ],
                                                            closingFollowingIndex: 94,
                                                            closingInitialIndex: 93,
                                                            inline: false,
                                                            innerFirstIndex: 85,
                                                            openingFollowingIndex: 85,
                                                            openingInitialIndex: 84,
                                                            outerFirstIndex: 93,
                                                            type: 'womTableCell',
                                                        },
                                                    ],
                                                    closingFollowingIndex: 95,
                                                    closingInitialIndex: 93,
                                                    inline: false,
                                                    innerFirstIndex: 84,
                                                    openingFollowingIndex: 85,
                                                    openingInitialIndex: 83,
                                                    outerFirstIndex: 95,
                                                    type: 'womTableRow',
                                                },
                                                {
                                                    attributes: {},
                                                    children: [],
                                                    closingFollowingIndex: 96,
                                                    closingInitialIndex: 96,
                                                    inline: true,
                                                    innerFirstIndex: 95,
                                                    openingFollowingIndex: 95,
                                                    openingInitialIndex: 95,
                                                    outerFirstIndex: 96,
                                                    type: 'unknown',
                                                },
                                            ],
                                            closingFollowingIndex: 98,
                                            closingInitialIndex: 96,
                                            inline: true,
                                            innerFirstIndex: 82,
                                            openingFollowingIndex: 82,
                                            openingInitialIndex: 80,
                                            outerFirstIndex: 98,
                                            type: 'womTable',
                                        },
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 99,
                                            closingInitialIndex: 99,
                                            inline: false,
                                            innerFirstIndex: 98,
                                            openingFollowingIndex: 98,
                                            openingInitialIndex: 98,
                                            outerFirstIndex: 99,
                                            type: 'unknown',
                                        },
                                    ],
                                    closingFollowingIndex: 100,
                                    closingInitialIndex: 99,
                                    inline: false,
                                    innerFirstIndex: 79,
                                    openingFollowingIndex: 79,
                                    openingInitialIndex: 78,
                                    outerFirstIndex: 99,
                                    type: 'womTableCell',
                                },
                                {
                                    attributes: {},
                                    children: [
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 101,
                                            closingInitialIndex: 101,
                                            inline: false,
                                            innerFirstIndex: 100,
                                            openingFollowingIndex: 100,
                                            openingInitialIndex: 100,
                                            outerFirstIndex: 101,
                                            type: 'unknown',
                                        },
                                        {
                                            type: 'womTable',
                                            inline: true,
                                            attributes: {},
                                            closingFollowingIndex: 119,
                                            closingInitialIndex: 117,
                                            innerFirstIndex: 103,
                                            openingFollowingIndex: 103,
                                            openingInitialIndex: 101,
                                            outerFirstIndex: 119,
                                            children: [
                                                {
                                                    type: 'unknown',
                                                    inline: true,
                                                    attributes: {},
                                                    openingInitialIndex: 103,
                                                    openingFollowingIndex: 103,
                                                    innerFirstIndex: 103,
                                                    closingInitialIndex: 104,
                                                    closingFollowingIndex: 104,
                                                    outerFirstIndex: 104,
                                                    children: [],
                                                },
                                                {
                                                    attributes: {},
                                                    children: [
                                                        {
                                                            attributes: {},
                                                            children: [
                                                                {
                                                                    attributes: {},
                                                                    children: [],
                                                                    closingFollowingIndex: 114,
                                                                    closingInitialIndex: 114,
                                                                    inline: false,
                                                                    innerFirstIndex: 106,
                                                                    openingFollowingIndex: 106,
                                                                    openingInitialIndex: 106,
                                                                    outerFirstIndex: 114,
                                                                    type: 'unknown',
                                                                },
                                                            ],
                                                            closingFollowingIndex: 115,
                                                            closingInitialIndex: 114,
                                                            inline: false,
                                                            innerFirstIndex: 106,
                                                            openingFollowingIndex: 106,
                                                            openingInitialIndex: 105,
                                                            outerFirstIndex: 114,
                                                            type: 'womTableCell',
                                                        },
                                                    ],
                                                    closingFollowingIndex: 116,
                                                    closingInitialIndex: 114,
                                                    inline: false,
                                                    innerFirstIndex: 105,
                                                    openingFollowingIndex: 106,
                                                    openingInitialIndex: 104,
                                                    outerFirstIndex: 116,
                                                    type: 'womTableRow',
                                                },
                                                {
                                                    attributes: {},
                                                    children: [],
                                                    closingFollowingIndex: 117,
                                                    closingInitialIndex: 117,
                                                    inline: true,
                                                    innerFirstIndex: 116,
                                                    openingFollowingIndex: 116,
                                                    openingInitialIndex: 116,
                                                    outerFirstIndex: 117,
                                                    type: 'unknown',
                                                },
                                            ],
                                        },
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 120,
                                            closingInitialIndex: 120,
                                            inline: false,
                                            innerFirstIndex: 119,
                                            openingFollowingIndex: 119,
                                            openingInitialIndex: 119,
                                            outerFirstIndex: 120,
                                            type: 'unknown',
                                        },
                                    ],
                                    closingFollowingIndex: 121,
                                    closingInitialIndex: 120,
                                    inline: false,
                                    innerFirstIndex: 100,
                                    openingFollowingIndex: 100,
                                    openingInitialIndex: 99,
                                    outerFirstIndex: 120,
                                    type: 'womTableCell',
                                },
                                {
                                    attributes: {},
                                    children: [
                                        {
                                            attributes: {},
                                            children: [],
                                            closingFollowingIndex: 129,
                                            closingInitialIndex: 129,
                                            inline: false,
                                            innerFirstIndex: 121,
                                            openingFollowingIndex: 121,
                                            openingInitialIndex: 121,
                                            outerFirstIndex: 129,
                                            type: 'unknown',
                                        },
                                    ],
                                    closingFollowingIndex: 130,
                                    closingInitialIndex: 129,
                                    inline: false,
                                    innerFirstIndex: 121,
                                    openingFollowingIndex: 121,
                                    openingInitialIndex: 120,
                                    outerFirstIndex: 129,
                                    type: 'womTableCell',
                                },
                            ],
                            closingFollowingIndex: 131,
                            closingInitialIndex: 129,
                            inline: false,
                            innerFirstIndex: 69,
                            openingFollowingIndex: 70,
                            openingInitialIndex: 68,
                            outerFirstIndex: 131,
                            type: 'womTableRow',
                        },
                        {
                            attributes: {},
                            children: [],
                            closingFollowingIndex: 132,
                            closingInitialIndex: 132,
                            inline: true,
                            innerFirstIndex: 131,
                            openingFollowingIndex: 131,
                            openingInitialIndex: 131,
                            outerFirstIndex: 132,
                            type: 'unknown',
                        },
                    ],
                    closingFollowingIndex: 135,
                    closingInitialIndex: 132,
                    inline: true,
                    innerFirstIndex: 3,
                    openingFollowingIndex: 3,
                    openingInitialIndex: 0,
                    outerFirstIndex: 135,
                    type: 'womTable',
                },
            ],
        ]
    );

    runSamples(parseInlineWomTable, samples);
});
