const { whitespace } = require('./is-markdown-whitespace');

function isLineFeedChar(value, index) {
    return value[index] === '\n';
}

function isWhitespaceChar(value, index) {
    return whitespace(value[index]);
}

// Это описано в спеке.
// Конец и начало строки для упрощения считаюся whitespace
function isCommonmarkWhitespace(value, index) {
    if (index < 0) {
        return true;
    }

    if (index >= value.length) {
        return true;
    }

    return isWhitespaceChar(value, index);
}

function isEscaped(value, index) {
    let i = index;

    while (i > 0 && value[i - 1] === '\\') {
        i -= 1;
    }

    return (index - i) % 2 === 1;
}

// ((url заголовок)) | [[url заголовок]] | [заголовок](url)
const R_HYPERLINK = /(?:\(\((?:[^\\]|\\.)+?\)\))|(?:\[\[(?:[^\\]|\\.)+?]])|(?:\[(?:[^\\\]]|\\.)+?]\((?:[^\\)]|\\.)+?\))/g;

// В remark есть баг, он не видит ссылки внутри строчных форматтеров
// https://github.com/remarkjs/remark/issues/459
// Поэтому находит внутри ссылок токены закрытия строчного форматтера,
function setNextLinkPosition(nextLink, value, fromIndex) {
    nextLink.opening = nextLink.closing = Infinity;

    R_HYPERLINK.lastIndex = fromIndex;

    const match = R_HYPERLINK.exec(value);

    if (match) {
        nextLink.opening = match.index;
        nextLink.closing = R_HYPERLINK.lastIndex;
    }
}

// https://spec.commonmark.org/0.29/#left-flanking-delimiter-run
function checkLeftFlankingDelimiterRunBounds(value, index, right, strict) {
    if (isCommonmarkWhitespace(value, right)) {
        return false;
    }

    // strict это по-сути костыль для emphasis, который через "_".
    // Который мешает нашему другому костылю для ссылок,
    // из за которого мы вынуждены заменить базовый emphasis на свой
    if (strict && !isCommonmarkWhitespace(value, index - 1)) {
        return false;
    }

    return true;
}

// https://spec.commonmark.org/0.29/#right-flanking-delimiter-run
function checkRightFlankingDelimiterRunBounds(value, index, right, strict) {
    if (isCommonmarkWhitespace(value, index - 1)) {
        return false;
    }

    if (strict && !isCommonmarkWhitespace(value, right)) {
        return false;
    }

    return true;
}

function findTailIndex(value, fromIndex, char, charTimes, strict) {
    const valueLength = value.length;
    // opening - начало ссылки
    // closing - конец ссылки
    const nextLink = {
        opening: 0,
        closing: 0,
    };

    for (let index = fromIndex; index < valueLength; index += 1) {
        if (index >= nextLink.closing) {
            // Известная ссылка находится позади текущего индекса,
            // находим новую, следующую
            setNextLinkPosition(nextLink, value, index);
        }

        if (index >= nextLink.opening && index < nextLink.closing) {
            // Текущий индекс находится внутри ссылки,
            // перескакиваем ссылку
            index = nextLink.closing - 1;

            continue;
        }

        let right = index;

        while (right < valueLength && value[right] === char) {
            right += 1;
        }

        if (right - index < charTimes) {
            index = right;

            continue;
        }

        if (checkRightFlankingDelimiterRunBounds(value, index, right, strict) && !isEscaped(value, index)) {
            return index;
        }

        index = right;
    }

    return -1;
}

exports.isCommonmarkWhitespace = isCommonmarkWhitespace;
exports.isWhitespaceChar = isWhitespaceChar;
exports.isLineFeedChar = isLineFeedChar;

exports.isEscaped = isEscaped;

exports.checkLeftFlankingDelimiterRunBounds = checkLeftFlankingDelimiterRunBounds;
exports.checkRightFlankingDelimiterRunBounds = checkRightFlankingDelimiterRunBounds;

exports.findTailIndex = findTailIndex;
