const constants = require('./constants');
const logger = require('./logger');
const utils = require('./utils');

/**
 * Определение необходимости повторного запроса и задержки перед его осуществлением.
 *
 * @param {number} retryNumber
 * @param {Object} error
 * @param {Object} options
 * @returns {number}
 */
function prepareRetry(retryNumber, error, options) {
    /**
    * При сетевых проблемах `got` не триггерит событие `response`.
    * Поэтому эмулируем ответ сервера, чтобы в логах была запись о проблемах сети.
    */
    if (utils.isResponseWithNetworkProblem(error) || utils.isTimedoutResponse(error)) {
        logger.logResponse(error, options);
    }

    const nextRetryDelay = getNextRetryDelay(error, options, retryNumber);

    logger.logRetry(error, options, retryNumber, nextRetryDelay);

    if (isSkippedRetry(error, options, retryNumber, nextRetryDelay)) {
        return 0;
    }

    return nextRetryDelay;
}

/**
 * @param {Object} error
 * @param {Object} options
 * @param {number} retryNumber
 * @returns {number}
 */
function getNextRetryDelay(error, options, retryNumber) {
    if (!options.ignoreRetryAfterHeader && error.headers && 'retry-after' in error.headers) {
        return extractRetryDelayFromHeader(error);
    }

    return utils.getBackoff(retryNumber);
}

/**
 * Извлечение информации из заголовка `retry-after`.
 *
 * По спецификации в заголовке может содержаться дата или секунды,
 * по прошествию которых нужно выполнить повторный запрос.
 *
 * @param {Object} error
 * @returns {number}
 */
function extractRetryDelayFromHeader(error) {
    const retryDelay = parseInt(error.headers['retry-after'], 10);

    if (isNaN(retryDelay)) {
        return Date.parse(error.headers['retry-after']) - Date.now();
    }

    return retryDelay * 1000;
}

/**
 * @param {Object} error
 * @param {Object} options
 * @param {number} retryNumber
 * @param {number} nextRetryDelay
 * @returns {boolean}
 */
function isSkippedRetry(error, options, retryNumber, nextRetryDelay) {
    if (isSkippedRetryByResponse(error, options)) {
        return true;
    }

    if (isSkippedRetryByMaxRetryCount(error, options, retryNumber)) {
        return true;
    }

    if (isSkippedRetryByMaxRetryDelay(error, options, nextRetryDelay)) {
        return true;
    }

    if (isSkippedRetryByMaxTotalRequestTime(error, options, nextRetryDelay)) {
        return true;
    }

    return false;
}

/**
 * @param {Object} error
 * @param {Object} options
 * @param {number} retryNumber
 * @returns {boolean}
 */
function isSkippedRetryByMaxRetryCount(error, options, retryNumber) {
    if (retryNumber <= options.retryCount) {
        return false;
    }

    logger.logSkippedRetryByMaxRetryCount(error, options);

    return true;
}

/**
 * @param {Object} error
 * @param {Object} options
 * @param {number} nextRetryDelay
 * @returns {boolean}
 */
function isSkippedRetryByMaxRetryDelay(error, options, nextRetryDelay) {
    if (nextRetryDelay <= options.retry.maxRetryAfter) {
        return false;
    }

    logger.logSkippedRetryByMaxRetryDelay(error, options, nextRetryDelay);

    return true;
}

/**
 * @param {Object} error
 * @param {Object} options
 * @param {number} nextRetryDelay
 * @returns {boolean}
 */
function isSkippedRetryByMaxTotalRequestTime(error, options, nextRetryDelay) {
    const hasRequestExpiresHeader = error.headers && 'x-request-expires' in error.headers;

    if (!hasRequestExpiresHeader) {
        return false;
    }

    const nextRetryTime = options.applyMaxRetryTotalTimeBeforeRetry ? nextRetryDelay : 0;
    const nextRetryDate = Date.now() + nextRetryTime;
    const currentDate = new Date(nextRetryDate);
    const expiresDate = new Date(error.headers['x-request-expires']);

    if (currentDate <= expiresDate) {
        return false;
    }

    logger.logSkippedRetryByMaxTotalRequestTime(error);

    return true;
}

/**
 * @param {Object} error
 * @param {Object} options
 * @returns {boolean}
 */
function isSkippedRetryByResponse(error, options) {
    const isRetriableNetworkError = error.code ? constants.RETRIABLE_NETWORK_ERROR_CODES.includes(error.code) : true;

    if (!isRetriableNetworkError) {
        logger.logSkippedRetryByNetworkError(error);

        return true;
    }

    const isRetriableMethod = error.method ? options.retry.methods.has(error.method) : true;

    if (!isRetriableMethod) {
        logger.logSkippedRetryByMethod(error);

        return true;
    }

    const isRetriableStatusCode = error.statusCode ? options.retry.statusCodes.has(error.statusCode) : true;

    if (!isRetriableStatusCode) {
        logger.logSkippedRetryByStatusCode(error);

        return true;
    }

    return false;
}

module.exports = {
    prepareRetry,

    getNextRetryDelay,

    isSkippedRetryByMaxRetryCount,
    isSkippedRetryByMaxRetryDelay,
    isSkippedRetryByMaxTotalRequestTime,
    isSkippedRetryByResponse
};
