import { ReactElement } from 'react';
import { renderToString } from 'react-dom/server';
import * as ReactAssets from './react-assets';
import { AdapterRenderResult } from '../../lib/taburet/packages/adapters/Adapter/Adapter';

/**
 * Определение базовых классов web4
 * на основе общепоисковой платформы taburet.
 */
import {
    IPushAssetsOptions,
    IAdapterConstructor,
    IAdapterOptions,
    IAdaptersBundle,
    IAssetsBundle,
    ITemplateRunnerOptions,
    IExperimentMeta,
    Adapter as TaburetAdapter,
    AdaptersRuntime as TaburetAdaptersRuntime,
    logger,
} from '../../lib/taburet';

import {
    ISerpSnippet,
    ISerpDocument,
    IPrivExternals,
    ISerpSupportedPlatforms,
    BaobabSubtree,
    IBaobabPlainLogNode,
    IConstructCounter,
    Construct,
} from '../../typings';

import { counterSetup, ICounterSetup } from '../../lib/counter/counter';
import { ISnippetContext } from '../../lib/Context/SnippetContext';
import { convertBaobabToPlainForm } from '../../components/Counter/Counter';

interface IReactAssetsOptions {
    main: IPushAssetsOptions;
    polyfill: IPushAssetsOptions;
}

/**
 * Параметры конструктора для базового класса адаптера web4.
 *
 * @param <S> поисковый сниппет для адаптера должен наследоваться от ISerpSnippet
 */
export interface ISerpAdapterOptions<S extends ISerpSnippet = ISerpSnippet>
    extends IAdapterOptions<ISnippetContext, S, ISerpDocument> {
    privExternals: IPrivExternals;
}

/**
 * Строгое определение базового класса адаптера для web4.
 *
 * @param <S> поисковый сниппет для адаптера должен наследоваться от ISerpSnippet
 */
export class Adapter<S extends ISerpSnippet> extends TaburetAdapter<ISnippetContext, S, ISerpDocument> {
    /**
     * Возможность добраться до старых функций-хелперов из пред. стека.
     * Не использовать без острой необходимости, а выносить хелперы
     * в отдельные модули и подключать по месту использования.
     *
     * @see https://st.yandex-team.ru/SERP-66673
     * @deprecated
     */
    protected privExternals: IPrivExternals;

    protected counterSetup: (counter: IConstructCounter) => ICounterSetup;

    protected logTree: BaobabSubtree = new Map();

    constructor(options: ISerpAdapterOptions<S>) {
        super(options);
        this.privExternals = options.privExternals;
        this.counterSetup = counterSetup(this.context, this.privExternals.Counter);
    }

    /**
     * Преобразует данные бэкенда в API конструктора
     */
    transform(): (Construct.IAdapterResult & Construct.IBlock) | null {
        return super.transform();
    }

    /**
     * Доставляет статику на страницу.
     * Основывается на прямом вызове функции
     * из privExternals.
     *
     * @param options
     */
    pushAssets(options: IPushAssetsOptions) {
        this.privExternals.pushAssets(options);
        this.privExternals.pushBundle('i-react-loader');
        this.privExternals.pushBundle('i-react-bus');

        if (this.context.expFlags.react_progress !== undefined) {
            this.privExternals.pushBundle('react-progress');
        }
    }

    /**
     * Доставляет сам react на страницу
     */
    pushReactAssets(options: IReactAssetsOptions) {
        // Если нужны полифилы, то react-with-dom.js не загружаем.
        // В этом случае сперва отработают полифилы, a react загрузится из i-react-loader'a
        if (!ReactAssets.needPolyfill(this.context) && !this.context.expFlags.react_lazy_load_and_hydrate) {
            this.privExternals.pushAssets(options.main);
        }

        if (ReactAssets.needPolyfill(this.context)) {
            this.privExternals.pushAssets(options.polyfill);
        }
    }

    getLogTree(): IBaobabPlainLogNode[] {
        return convertBaobabToPlainForm(this.logTree);
    }

    getLogHash() {
        // При обработке ajax'а this.document может быть undefined.
        // В этом случае num=0 правильное значение.
        const num = (this.document && this.document.num) || 0;
        // TODO https://st.yandex-team.ru/SERP-76433
        return -100 * this.context.reportData.log.hash * (Number(num) + 1);
    }
}

/**
 * Адаптер, возвращающий конструкторский API в реестре.
 */
class CoreAdapterImplementation extends Adapter<ISerpSnippet> {}

const log = logger('serp:taburet:adapter:runtime');

/**
 * Строгое определение рантайма адаптеров для web4.
 */
export class AdaptersRuntime extends TaburetAdaptersRuntime<ISerpAdapterOptions<ISerpSnippet>, {}> {
    protected defaultBaseAdapter = Adapter;

    protected tryRender(instance: CoreAdapterImplementation): AdapterRenderResult | null {
        const renderResult = super.tryRender(instance);

        if (renderResult) {
            // ПОРЯДОК ВЫЗОВОВ ВАЖЕН:
            // - генерируется баобабное дерево
            const renderedString = renderToString(renderResult as ReactElement<{}>);
            // - получаем его через instance
            const logTree = instance.getLogTree();

            // - передаем дальше в приваки
            return this.wrapInConstructBlock(renderedString, logTree);
        }

        return null;
    }

    protected pushAssets(
        options: IAdapterOptions,
        adapterClass: IAdapterConstructor,
        instance: CoreAdapterImplementation,
        experiment: IExperimentMeta = {},
    ): void {
        instance.pushReactAssets({
            main: ReactAssets.mainAsset,
            polyfill: super.buildAssetOptions(options, ...ReactAssets.polyfillAsset),
        });

        super.pushAssets(options, adapterClass, instance, experiment);
    }

    /**
     * Заворачивает результат рендера в конструкторский блок html.
     * Отключаем заворачивание в legacyAdapter и запуск адаптеров
     * следом за результатом выполнения текущего.
     */
    private wrapInConstructBlock(content: string, logTree: IBaobabPlainLogNode[]): Construct.IReactCompatible {
        log.debug('Заворачиваем результат рендера в конструкторский блок', content);

        return {
            block: 'html',
            preventLegacy: true,
            godMode: true,
            content: {
                tag: '',
                logSubtree: logTree,
                content,
            },
        };
    }
}

/**
 * Параметры установки табурета.
 */
export interface ITaburetSetupOptions {
    adapters: AllPlatformsAdaptersBundle;
    assets?: AllPlatformsAssetsBundle;
    autobundlePath?: string;
}

/**
 * Параметры запуска табурета.
 */
export interface ITaburetRunOptions extends ITemplateRunnerOptions {
    /**
     * Название текущей платформы.
     */
    platform: keyof ISerpSupportedPlatforms;
}

/**
 * Бандл адаптеров для всех платформ.
 */
export type AllPlatformsAdaptersBundle = Record<keyof ISerpSupportedPlatforms, IAdaptersBundle>;

/**
 * Бандл статики для всех платформ.
 */
export type AllPlatformsAssetsBundle = Record<keyof ISerpSupportedPlatforms, IAssetsBundle>;
