import * as React from 'react';
import { cn } from '@bem-react/classname';
import {
    Textinput as TextinputPresenter,
    withHasClear,
    withThemeWebsearch,
    withViewClassic,
} from '@yandex-lego/components/Textinput/desktop';
import { compose, IClassNameProps } from '@bem-react/core';
import i18nFactory from '@yandex-int/i18n';

import { ISuggestResponse } from './Suggest.typings';
import { SuggestNetworker, ISuggestAPIParams } from './Networker/Suggest-Networker';
import * as keyset from './Suggest.i18n';

import './Suggest.scss';

const i18n = i18nFactory(keyset);

export interface ISuggestProps extends IClassNameProps, ISuggestAPIParams {
    /** Текст запроса */
    query?: string;
    /** Обработчик на выбор текста в саджесте */
    onSubmit?: (query: string) => void;
    /** С логотипом */
    withLogo?: boolean;
}

export interface ISuggestState {
    text: string;
    focused: boolean;
    suggestItems: string[];
    /** Индекс текущего выбраного элемента саджеста */
    currentItemIndex: number;
}

const Textinput = compose(withHasClear, withThemeWebsearch, withViewClassic)(TextinputPresenter);

export const cnSuggest = cn('Suggest');

export class Suggest extends React.PureComponent<ISuggestProps, ISuggestState> {
    constructor(props: ISuggestProps) {
        super(props);

        this.onFocus = this.onFocus.bind(this);
        this.onKeyDownHandler = this.onKeyDownHandler.bind(this);
        this.onSuggestItemClick = this.onSuggestItemClick.bind(this);
        this.submit = this.submit.bind(this);
        this.requestSuggestItems = this.requestSuggestItems.bind(this);
        this.chooseSuggestItem = this.chooseSuggestItem.bind(this);
        this.onDocumentClick = this.onDocumentClick.bind(this);
        this.blurSuggest = this.blurSuggest.bind(this);
    }
    private inputRef = React.createRef<HTMLInputElement>();
    private networker = new SuggestNetworker();

    static defaultProps = {
        withLogo: true,
    };

    state = {
        text: this.props.query || '',
        focused: false,
        suggestItems: [],
        currentItemIndex: 0,
    };

    protected requestSuggestItems() {
        if (this.state.text === '') {
            return;
        }

        this.networker.ask(
            {
                query: this.state.text,
                lr: this.props.lr,
                yandexuid: this.props.yandexuid,
                tld: this.props.tld,
            },
            {
                successCallback: this.onSuggestLoadingSuccess(),
                failureCallback: () => {},
            },
        );
    }

    protected onSuggestLoadingSuccess(cb?: () => void) {
        return (response: ISuggestResponse) => {
            this.setState(
                {
                    suggestItems: response[1].map(responseItem => responseItem[1]),
                    currentItemIndex: 0,
                    focused: true,
                },
                cb,
            );
        };
    }

    onChangeHandler = (e: React.ChangeEvent<HTMLInputElement>) => {
        this.setState({ text: e.target.value }, this.requestSuggestItems);
    };

    onFocus() {
        this.setState({ focused: true }, this.requestSuggestItems);
    }

    /** Выполняем запрос */
    submit() {
        if (this.state.text === '') {
            return;
        }

        this.props.onSubmit && this.props.onSubmit(this.state.text);
    }

    onKeyDownHandler(e: React.KeyboardEvent<HTMLInputElement>) {
        const { currentItemIndex, suggestItems } = this.state;

        if (e.keyCode === 13 && this.props.onSubmit) {
            if (currentItemIndex > 0 && currentItemIndex <= suggestItems.length + 1) {
                this.chooseSuggestItem(suggestItems[currentItemIndex - 1]);
            }

            this.submit();
            this.blurSuggest();
        }

        /** ARROW DOWN */
        if (e.keyCode === 40) {
            this.setNextActiveItem();
        }

        /** ARROW UP */
        if (e.keyCode === 38) {
            this.setPreviousActiveItem();
        }

        /** ESC */
        if (e.keyCode === 27) {
            this.blurSuggest();
        }
    }

    protected setNextActiveItem() {
        const mod = this.state.suggestItems.length + 1;
        this.setState({
            currentItemIndex: (this.state.currentItemIndex + 1) % mod,
        });
    }

    protected setPreviousActiveItem() {
        const mod = this.state.suggestItems.length + 1;
        this.setState({
            currentItemIndex: (mod + this.state.currentItemIndex - 1) % mod,
        });
    }

    /** Выбираем элемент саджеста */
    protected chooseSuggestItem(text: string) {
        this.setState(
            {
                text,
                suggestItems: [],
                focused: false,
            },
            this.submit,
        );
    }

    onSuggestItemClick(item: string) {
        return (e: React.MouseEvent) => {
            e.preventDefault();
            e.stopPropagation();

            this.chooseSuggestItem(item);
        };
    }

    renderSuggestItems() {
        const hovered = (index: number) => this.state.currentItemIndex === index + 1;

        return (
            <div className={cnSuggest('Group')}>
                {this.state.suggestItems.map((item, index) => (
                    <div key={item} className={cnSuggest('ItemWrap')}>
                        <div
                            onClick={this.onSuggestItemClick(item)}
                            className={cnSuggest('Item', { hovered: hovered(index) })}
                        >
                            {item}
                        </div>
                    </div>
                ))}
            </div>
        );
    }

    blurSuggest() {
        this.inputRef.current && this.inputRef.current.blur();
        this.setState({
            focused: false,
            suggestItems: [],
        });
    }

    onDocumentClick(e: MouseEvent) {
        /** Прячем саджест, имитируя onBlur */
        if (this.state.focused !== false && (e.target as HTMLInputElement).className !== 'YTextinput-Control') {
            this.blurSuggest();
        }
    }

    componentDidMount() {
        document.addEventListener('click', this.onDocumentClick);
    }

    componentWillUnmount() {
        document.removeEventListener('click', this.onDocumentClick);
    }

    render() {
        const { text, focused } = this.state;
        const { className, withLogo } = this.props;

        return (
            <div className={cnSuggest({ focused }, [className])}>
                {withLogo && <div className={cnSuggest('Logo')} />}
                <div className={cnSuggest('Wrap')}>
                    <Textinput
                        hasClear
                        theme="websearch"
                        view="classic"
                        value={text}
                        autoComplete="off"
                        name="text"
                        onFocus={this.onFocus}
                        onChange={this.onChangeHandler}
                        controlRef={this.inputRef}
                        onKeyDown={this.onKeyDownHandler}
                    />
                    <button className={cnSuggest('Button')} onClick={this.submit}>
                        <div className={cnSuggest('ButtonText')}>{i18n('Найти')}</div>
                    </button>
                </div>
                {focused && this.renderSuggestItems()}
            </div>
        );
    }
}
