import "jest-extended";
import { ErrorResponse, Secret, VaultClient, VersionResponse } from "./index";

import requests from "@yandex-int/si.ci.requests";
jest.mock("@yandex-int/si.ci.requests");

const requestsStub = (requests as unknown) as jest.Mock<
    requests.GotPromise<string>,
    [requests.GotUrl, requests.GotOptions<string>]
>;

function fakeResponse(value: Array<Secret>): VersionResponse {
    return {
        status: "ok",
        version: {
            comment: "",
            created_at: 1,
            created_by: 1,
            creator_login: "tester",
            secret_name: "test secret name",
            secret_uuid: "some uuid",
            value,
            version: "1",
        },
    };
}

function mockResponse(value: Array<Secret>): void {
    const response = ({ body: JSON.stringify(fakeResponse(value)) } as unknown) as requests.Response<string>;
    requestsStub.mockResolvedValue(response);
}

function mockError(code: string, message: string): void {
    const value: ErrorResponse = {
        status: "error",
        message,
        code,
    };
    const response = ({ body: JSON.stringify(value) } as unknown) as requests.Response<string>;
    requestsStub.mockResolvedValue(response);
}

describe("VaultClient", () => {
    describe("getVersion", () => {
        it("should use port from urlParams", async () => {
            mockResponse([]);
            const client = new VaultClient("test", { port: "17" });
            await client.getVersion("testSecret");
            expect(requestsStub).toBeCalledWith(expect.objectContaining({ port: "17" }), expect.anything());
        });

        it("should use host from urlParams", async () => {
            mockResponse([]);
            const client = new VaultClient("test", { host: "some.test.host" });
            await client.getVersion("testSecret");
            expect(requestsStub).toBeCalledWith(expect.objectContaining({ host: "some.test.host" }), expect.anything());
        });

        it("should append authorization header", async () => {
            mockResponse([]);
            const client = new VaultClient("test");
            await client.getVersion("testSecret");
            expect(requestsStub).toBeCalledWith(
                expect.anything(),
                expect.objectContaining({ headers: { Authorization: "OAuth test" } }),
            );
        });

        it("should make request to correct pathname", async () => {
            mockResponse([]);
            const client = new VaultClient("test");
            await client.getVersion("testSecret");
            expect(requestsStub).toBeCalledWith(
                expect.objectContaining({ pathname: "/1/versions/testSecret" }),
                expect.anything(),
            );
        });

        it("should throw an error if vault responded with error", async () => {
            mockError("418", "I'm a teapot");
            const client = new VaultClient("test");
            await expect(client.getVersion("testSecret")).rejects.toThrow("418: I'm a teapot");
        });

        it("should return secret value", async () => {
            const value = [{ key: "token", value: "abcdef" }];
            mockResponse(value);
            const client = new VaultClient("test");
            const res = await client.getVersion("testSecret");
            expect(res).toStrictEqual(value);
        });
    });
});
