import requests from "@yandex-int/si.ci.requests";
import { URL } from "url";

type CommonResponsePart = {
    status: "error" | "ok";
};

type RawResponse<T> = ErrorResponse | T;

function resultIsError<T extends CommonResponsePart>(response: RawResponse<T>): response is ErrorResponse {
    return response.status === "error";
}

/**
 * Поля объекта URL, которые библиотека позволяет переопределить при запросах к серверу
 */
type UrlParams = Partial<URL>;

const defaultVaultHost = "https://vault-api.passport.yandex.net/";

// При использовании `requestOptions?: Partial<...>` непосредственно в конструкторе возникает ошибка TS2742: The inferred type of 'requestOptions' cannot be named without a reference to '@yandex-int/si.ci.requests/node_modules/@types/got'. This is likely not portable. A type annotation is necessary
// @see https://github.com/microsoft/TypeScript/issues/30858
type PartialOptions = Partial<requests.GotOptions<string>> | undefined;

export type Secret = {
    key: string;
    value: string;
};

export type ErrorResponse = {
    status: "error";
    message: string;
    code: string;
};

export type VersionResponse = {
    status: "ok";
    version: {
        comment: string;
        created_at: number;
        created_by: number;
        creator_login: string;
        secret_name: string;
        secret_uuid: string;
        value: Array<Secret>;
        version: string;
    };
};

/**
 * В текущей версии поддерживается авторизация только по OAuth-токену.
 * @see https://vault-api.passport.yandex.net/docs/
 * @see https://a.yandex-team.ru/arc/trunk/arcadia/library/python/vault_client
 */
export class VaultClient {
    constructor(
        protected oAuthToken: string,
        protected urlParams?: UrlParams,
        protected requestOptions?: PartialOptions,
    ) {}

    /**
     * Возвращает значение секрета
     * @param id - идентификатор версии либо идентификатор секрета. Если указан идентификатор секрета, будет возвращен
     * секрет последней версии.
     */
    async getVersion(id: string): Promise<Array<Secret>> {
        return this.request(`/1/versions/${id}`);
    }

    protected async request(pathname: string): Promise<Array<Secret>> {
        const url = new URL(defaultVaultHost);
        const options = {
            ...this.requestOptions,
            headers: {
                ...(this.requestOptions || {}).headers,
                Authorization: `OAuth ${this.oAuthToken}`,
            },
        };

        Object.assign(url, this.urlParams, { pathname });

        const res = await requests(url, options);

        // TODO: добавить валидацию возвращаемого ответа вместо простого as
        const response = JSON.parse(res.body) as RawResponse<VersionResponse>;
        if (resultIsError(response)) {
            throw new Error(`${response.code}: ${response.message}`);
        }

        return response.version.value;
    }
}
