"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var path = require("path");
var ts = require("typescript");
var lineBreakRegex = /\r\n|\r|\n/;
var lineBreakRegexGlobal = /\r\n|\r|\n/g;
function getLines(input) {
    return input.split(lineBreakRegex);
}
function getLineInfo(code) {
    var lines = [];
    lineBreakRegexGlobal.lastIndex = 0;
    var lastOffset = 0;
    var match;
    var lineNumber = 0;
    while ((match = lineBreakRegexGlobal.exec(code)) !== null) {
        lines.push({
            number: ++lineNumber,
            offset: lastOffset,
            text: code.substring(lastOffset, match.index),
            lineBreak: match[0],
        });
        lastOffset = match.index + match[0].length;
    }
    lines.push({ number: ++lineNumber, offset: lastOffset, text: code.substr(lastOffset), lineBreak: null });
    return lines;
}
function getNodeRange(node) {
    if (typeof node.getStart === 'function' &&
        typeof node.getEnd === 'function') {
        return [node.getStart(), node.getEnd()];
    }
    else if (typeof node.pos !== 'undefined' &&
        typeof node.end !== 'undefined') {
        return [node.pos, node.end];
    }
    return [0, 0];
}
function extractKeyFromCall(callExpression) {
    var key;
    if (callExpression.arguments.length) {
        var firstArg = callExpression.arguments[0];
        if (firstArg.kind === ts.SyntaxKind.StringLiteral) {
            var keyLiteral = firstArg;
            key = {
                name: keyLiteral.text,
                nameRaw: keyLiteral.getText(),
                range: getNodeRange(callExpression),
                location: { line: 0, column: 0 },
                fragment: '',
                keyset: '',
            };
        }
        var secondArg = callExpression.arguments[1];
        if (secondArg && secondArg.kind === ts.SyntaxKind.ObjectLiteralExpression) {
            var optionsExpression = secondArg;
            optionsExpression.properties.forEach(function (prop) {
                var _a;
                if (ts.isPropertyAssignment(prop)) {
                    var assignment = prop;
                    var initializerText = prop.initializer.kind === ts.SyntaxKind.StringLiteral ?
                        prop.initializer.text :
                        prop.initializer.getText();
                    var paramName = assignment.name.getText();
                    if (key) {
                        if (paramName === 'context') {
                            key.context = initializerText;
                        }
                        else {
                            key.params || (key.params = []);
                            key.params.push((_a = {},
                                _a[paramName] = initializerText,
                                _a));
                        }
                    }
                }
            });
        }
    }
    return key;
}
function isI18nCall(callExpression, i18nIdentifierName) {
    return callExpression.expression.getText() === i18nIdentifierName;
}
function isI18nWithKeysetsCall(callExpression, i18nWithKeysetsIdentifierName) {
    return callExpression.expression.getText() === i18nWithKeysetsIdentifierName;
}
function getRequiredI18nIdentifierName(requiredVarDecl, i18nModuleName) {
    if (requiredVarDecl.initializer && requiredVarDecl.initializer.kind === ts.SyntaxKind.CallExpression) {
        var callExpression = requiredVarDecl.initializer;
        if (callExpression.expression.getText() === 'require' &&
            callExpression.arguments.length === 1 &&
            callExpression.arguments[0].getText() === "'" + i18nModuleName + "'") {
            return requiredVarDecl.name.getText();
        }
    }
    return '';
}
function getLocationOfFragment(sourceCode, fragment) {
    var linesInfo = getLineInfo(sourceCode);
    var fragmentLines = getLines(fragment);
    var fragmentStart = sourceCode.indexOf(fragment);
    for (var i = 0; ~fragmentStart && i < linesInfo.length - 1; i++) {
        var info = linesInfo[i];
        if (info.offset === fragmentStart) {
            return { line: info.number, column: 0 };
        }
        if (info.offset > fragmentStart) {
            var line = info.number - 1;
            var prevLineInfo = linesInfo[i - 1];
            var column = prevLineInfo.text.indexOf(fragmentLines[0]);
            return { line: line, column: column };
        }
    }
    return { line: 0, column: 0 };
}
function getKeySetName(importDecl, i18nKeysetExtension) {
    var moduleNameWithQuotes = importDecl.moduleSpecifier.getText();
    var moduleName = moduleNameWithQuotes.substring(1, moduleNameWithQuotes.length - 1);
    var reg = new RegExp("\\." + i18nKeysetExtension + "$");
    if (reg.test(moduleName)) {
        return path.basename(moduleName.replace(reg, ''));
    }
    return '';
}
function getImportI18nIdentifierName(importDecl, i18nModuleName, keyName) {
    var moduleNameWithQuotes = importDecl.moduleSpecifier.getText();
    var moduleName = moduleNameWithQuotes.substring(1, moduleNameWithQuotes.length - 1);
    var reg = new RegExp(i18nModuleName + "$");
    if (reg.test(moduleName) && importDecl.importClause) {
        var identifier = void 0;
        if (keyName === 'default') {
            identifier = importDecl.importClause.name ||
                importDecl.importClause.namedBindings.name;
        }
        else {
            var namedBindings = importDecl.importClause.namedBindings &&
                importDecl.importClause.namedBindings.elements;
            if (namedBindings && namedBindings.length) {
                identifier = (namedBindings.find(function (item) {
                    var name = item.propertyName && item.propertyName.text || item.name.text;
                    return name === keyName;
                }));
                identifier = identifier && identifier.name;
            }
        }
        return identifier ? identifier.text : '';
    }
    return '';
}
function extractI18NKeysFromSourceFile(sourceFile, i18nModuleName, i18nKeysetExtension) {
    if (i18nModuleName === void 0) { i18nModuleName = 'bem-i18n'; }
    if (i18nKeysetExtension === void 0) { i18nKeysetExtension = 'i18n'; }
    var keys = [];
    var keySetName = '';
    var i18nIdentifierName;
    var i18nWithKeysetsIdentifierName;
    var currentKeyName;
    function next(node) {
        node.forEachChild(walk);
    }
    function walk(node) {
        var firstLevel = node.parent && node.parent.kind === ts.SyntaxKind.SourceFile;
        if (firstLevel && node.kind === ts.SyntaxKind.ImportDeclaration) {
            !i18nIdentifierName &&
                (i18nIdentifierName = getImportI18nIdentifierName(node, i18nModuleName, currentKeyName));
            !keySetName &&
                (keySetName = getKeySetName(node, i18nKeysetExtension));
        }
        if (node.kind === ts.SyntaxKind.VariableDeclaration) {
            !i18nIdentifierName &&
                (i18nIdentifierName = getRequiredI18nIdentifierName(node, i18nModuleName));
        }
        if (node.kind === ts.SyntaxKind.CallExpression) {
            var callExpression = node;
            if (i18nIdentifierName &&
                isI18nCall(callExpression, i18nIdentifierName)) {
                var parent = callExpression.parent;
                if (!parent) {
                    return next(node);
                }
                if (parent.kind === ts.SyntaxKind.VariableDeclaration) {
                    var varDecl = callExpression.parent;
                    var identifier = varDecl.name;
                    i18nWithKeysetsIdentifierName = identifier.getText();
                    return next(node);
                }
                if (parent.kind === ts.SyntaxKind.CallExpression) {
                    var i18nKey = extractKeyFromCall(callExpression.parent);
                    i18nKey && keys.push(i18nKey);
                    return next(node);
                }
            }
            if (i18nWithKeysetsIdentifierName &&
                isI18nWithKeysetsCall(callExpression, i18nWithKeysetsIdentifierName)) {
                var i18nKey = extractKeyFromCall(callExpression);
                i18nKey && keys.push(i18nKey);
                return next(node);
            }
        }
        return next(node);
    }
    ts.bindSourceFile(sourceFile, {
        target: ts.ScriptTarget.ES5, module: ts.ModuleKind.CommonJS,
    });
    var text = sourceFile.text;
    ['default', 'i18nRaw'].forEach(function (keyName) {
        currentKeyName = keyName;
        i18nIdentifierName = '';
        i18nWithKeysetsIdentifierName = '';
        ts.forEachChild(sourceFile, walk);
    });
    keys.forEach(function (key) {
        key.fragment = text.substring(key.range[0], key.range[1]);
        key.location = getLocationOfFragment(text, key.fragment);
        key.keyset = keySetName;
        key.params && key.params.length || (key.params = undefined);
    });
    return keys;
}
function extractI18NKeysFromFiles(fileNames, i18nModuleName, i18nKeysetExtension) {
    var keys = [];
    var program = ts.createProgram(fileNames, {
        target: ts.ScriptTarget.ES5, module: ts.ModuleKind.CommonJS,
    });
    program.getSourceFiles().forEach(function (sourceFile) {
        if (!sourceFile.isDeclarationFile) {
            keys = keys.concat(extractI18NKeysFromSourceFile(sourceFile, i18nModuleName, i18nKeysetExtension));
        }
    });
    return keys;
}
exports.extractI18NKeysFromFiles = extractI18NKeysFromFiles;
function extractI18NKeysFromSourceCode(sourceCode, i18nModuleName, i18nKeysetExtension, sourceFileName) {
    if (sourceFileName === void 0) { sourceFileName = 'sourceCode.tsx'; }
    var sourceFile = ts.createSourceFile(sourceFileName, sourceCode, ts.ScriptTarget.ES5, true);
    return extractI18NKeysFromSourceFile(sourceFile, i18nModuleName, i18nKeysetExtension);
}
exports.extractI18NKeysFromSourceCode = extractI18NKeysFromSourceCode;
