'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var React = require('react');

/* global Reflect, Promise */

var __assign = function() {
    __assign = Object.assign || function __assign(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p)) t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};

var registryContext = React.createContext({});
var RegistryProvider = registryContext.Provider;
var RegistryConsumer = registryContext.Consumer;
function withRegistry() {
    // Use arguments instead of rest-arguments to get faster and more compact code.
    var registries = [].slice.call(arguments);
    return function WithRegistry(Component) {
        var RegistryResolver = function (props) {
            var providedRegistriesRef = React.useRef(null);
            return (React.createElement(RegistryConsumer, null, function (contextRegistries) {
                if (providedRegistriesRef.current === null) {
                    var providedRegistries = __assign({}, contextRegistries);
                    for (var i = 0; i < registries.length; i++) {
                        var registry = registries[i];
                        var overrides = contextRegistries[registry.id];
                        // eslint-disable-next-line no-nested-ternary
                        providedRegistries[registry.id] = registry.overridable
                            ? overrides
                                ? registry.merge(overrides)
                                : registry
                            : registry && overrides
                                ? overrides.merge(registry)
                                : registry;
                    }
                    providedRegistriesRef.current = providedRegistries;
                }
                return (React.createElement(RegistryProvider, { value: providedRegistriesRef.current }, React.createElement(Component, props)));
            }));
        };
        {
            var resolverValue = registries.map(function (registry) { return registry.id; }).join(', ');
            // TODO: Use setDisplayName util.
            RegistryResolver.displayName = "RegistryResolver(" + resolverValue + ")";
        }
        return RegistryResolver;
    };
}
var ComponentRegistryConsumer = function (props) { return (React.createElement(RegistryConsumer, null, function (registries) {
    {
        if (!registries[props.id]) {
            throw new Error("Registry with id '" + props.id + "' not found.");
        }
    }
    return props.children(registries[props.id].snapshot());
})); };
var useRegistries = function () {
    return React.useContext(registryContext);
};
var useComponentRegistry = function (id) {
    var registries = useRegistries();
    return registries[id].snapshot();
};
var registryHocMark = 'RegistryHoc';
function withBase(hoc) {
    var fakeComponent = function () {
        throw new Error("Not found base component for enhance HOC: " + hoc.toString());
    };
    fakeComponent.$symbol = registryHocMark;
    fakeComponent.hoc = hoc;
    return fakeComponent;
}
function isHoc(component) {
    return component.$symbol === registryHocMark;
}
var Registry = /** @class */ (function () {
    function Registry(_a) {
        var id = _a.id, _b = _a.overridable, overridable = _b === void 0 ? true : _b;
        this.components = {};
        this.id = id;
        this.overridable = overridable;
    }
    /**
     * Set react component in registry by id.
     *
     * @param id component id
     * @param component valid react component
     */
    Registry.prototype.set = function (id, component) {
        this.components[id] = component;
        return this;
    };
    /**
     * Set hoc for extends component in registry by id
     *
     * @param id component id
     * @param hoc hoc for extends component
     */
    Registry.prototype.extends = function (id, hoc) {
        this.components[id] = withBase(hoc);
        return this;
    };
    /**
     * Set react components in registry via object literal.
     *
     * @param componentsSet set of valid react components
     */
    Registry.prototype.fill = function (componentsSet) {
        for (var key in componentsSet) {
            this.components[key] = componentsSet[key];
        }
        return this;
    };
    /**
     * Get react component from registry by id.
     *
     * @param id component id
     */
    Registry.prototype.get = function (id) {
        {
            if (!this.components[id]) {
                throw new Error("Component with id '" + id + "' not found.");
            }
        }
        return this.components[id];
    };
    /**
     * Returns list of components from registry.
     */
    Registry.prototype.snapshot = function () {
        return this.components;
    };
    /**
     * Override components by external registry.
     * @internal
     *
     * @param otherRegistry external registry
     */
    Registry.prototype.merge = function (otherRegistry) {
        var clone = new Registry({ id: this.id, overridable: this.overridable });
        clone.fill(this.components);
        if (!otherRegistry)
            return clone;
        var otherRegistryComponents = otherRegistry.snapshot();
        for (var componentName in otherRegistryComponents) {
            if (!otherRegistryComponents.hasOwnProperty(componentName))
                continue;
            clone.components[componentName] = this.mergeComponents(clone.components[componentName], otherRegistryComponents[componentName]);
        }
        return clone;
    };
    /**
     * Returns extended or replacing for base impleme
     *
     * @param base base implementation
     * @param overrides overridden implementation
     */
    Registry.prototype.mergeComponents = function (base, overrides) {
        if (isHoc(overrides)) {
            if (!base)
                return overrides;
            if (isHoc(base)) {
                // If both components are hocs, then create compose-hoc
                return withBase(function (Base) { return overrides.hoc(base.hoc(Base)); });
            }
            return overrides.hoc(base);
        }
        return overrides;
    };
    return Registry;
}());

exports.ComponentRegistryConsumer = ComponentRegistryConsumer;
exports.Registry = Registry;
exports.RegistryConsumer = RegistryConsumer;
exports.registryContext = registryContext;
exports.useComponentRegistry = useComponentRegistry;
exports.useRegistries = useRegistries;
exports.withRegistry = withRegistry;
