let assert = require('assert');
let request = require('request');
let _ = require('lodash');
let vow = require('vow');
let xml2js = require('xml2js');
let resources = require('./resources');
let cache = {};

function getRejectMessage(description, p) {
    return [
        description,
        'Ручка: ' + p[0],
        'Параметры: ' + JSON.stringify(p[1], null, 4),
        'Ответ: ' + p[2],
    ].join('\n');
}

exports.resource = function(resource) {
    let deferred = vow.defer();
    let config = _.extend.apply(_, [{}].concat(_.toArray(arguments).slice(1)));
    let options = resources[resource](config);
    let cacheKey = JSON.stringify(options);

    if (config.cache !== false && cache[cacheKey]) {
        deferred.resolve(cache[cacheKey]);
    } else {
        request(options, function(error, response, body) {
            let params = [resource, options, body];

            if (error !== null) {
                deferred.reject(getRejectMessage('Не удалось получить данные.', params));
            } else if (/^\s*<\?xml/.test(body)) {
                // XML -> JSON.
                xml2js.parseString(body, function(error, result) {
                    if (error !== null) {
                        deferred.reject(
                            getRejectMessage(
                                'Не удалось преобразовать XML-данные в JSON.',
                                params
                            )
                        );
                    } else {
                        deferred.resolve([body, result]);
                    }
                });
            } else {
                // NONAME -> JSON.
                try {
                    deferred.resolve([body, JSON.parse(body)]);
                } catch (e) {
                    deferred.reject(
                        getRejectMessage(
                            'Не удалось преобразовать полученные данные в JSON.',
                            params
                        )
                    );
                }
            }
        });
    }

    return deferred.promise().spread(function(raw, json) {
        let error = json.error || (json.result || {}).error;

        (config.cache !== false) && (cache[cacheKey] = [raw, json]);
        (config.check !== false) && assert(!error, JSON.stringify(error));

        return config.raw ? raw : json;
    });
};

exports.head = function(config) {
    let base = _.pick(config, ['host', 'project', 'branch', 'token']);
    let additions = { cache: false, limit: 1 };

    return this
        .resource('/history/', base, additions)
        .then(function(response) {
            return response.commits[0].sha1;
        });
};

exports.keysets = function(config) {
    config = _.pick(config, ['host', 'project', 'hash', 'token']);

    return this
        .resource('/admin/project/short-name/keysets/', config)
        .then(function(response) {
            return _.pluck(response.data.items, 'name');
        });
};

exports.headKeysets = function(config) {
    let base = _.pick(config, ['host', 'project', 'branch', 'token']);
    let additions = { cache: false };

    return this
        .resource('/admin/project/short-name/keysets/', base, additions)
        .then(function(response) {
            return _.pluck(response.data.items, 'name');
        });
};

exports.addKeyset = function(config, keyset) {
    let base = _.pick(config, ['host', 'project', 'branch', 'token']);
    let additions = { cache: false, keyset: keyset };

    return this.resource('/admin/project/short-name/keyset/', base, additions);
};

exports.project = function(config) {
    config = _.pick(config, ['host', 'project', 'hash', 'token']);

    return this.resource('/admin/project/short-name/', config);
};

exports.language = function(config) {
    return this.project(config).then(function(response) {
        return response.data.original_language;
    });
};

exports.translations = function(config) {
    return this.resource('/projects/export/json/', config);
};

exports.export = function(config, hash) {
    let additions = { hash: hash, raw: true };

    return this.resource('/projects/export/' + config.pull + '/', config, additions);
};
