module.exports = function(program) {
    (program)
        .command('init')
        .description('Initialize tanker project in current directory')
        .action(function() {
            let fs = require('fs');
            let path = require('path');
            let async = require('async');
            let _ = require('lodash');
            let util = require('../lib/util');
            let print = util.print;
            let isYes = util.isYes;
            let TANKER_DIR = path.join(process.cwd(), '.tanker');
            let TANKER_CNF = path.join(process.cwd(), '.tanker/config.js');
            let config = {
                tanker: {},
                finders: [],
            };
            let tmpl = _.template([
                'module.exports = function(config) {',
                '',
                '<% _.each(tanker, function(val, key) { %>' +
                '    config.tanker.${ key } = \'${ val }\';',
                '<% }); %>',
                '    config.finders = [',
                '<% _.each(finders, function(val) { %>' +
                '        \'${ val }\',',
                '<% }); %>' +
                '    ];',
                '};',
                '',
            ].join('\n'));

            print.info('Please specify data for project config');

            async.series([
                function(step) {
                    if (fs.existsSync(TANKER_CNF)) {
                        print.info('Project config already exists in ' + TANKER_CNF);

                        program.prompt('Rewrite? ', function(answer) {
                            isYes(answer) ? step() : step('!');
                        });
                    } else {
                        step();
                    }
                },
                function(step) {
                    program.prompt('Project name in Tanker: ', function(answer) {
                        config.tanker.project = answer;
                        step();
                    });
                },
                function(step) {
                    program.prompt('Tanker authorization OAuth token: ', function(answer) {
                        config.tanker.token = answer;
                        step();
                    });
                },
                function(step) {
                    print.info('Specify one or more console commands for file search.');

                    async.forever(function(more) {
                        program.prompt('Command: ', function(answer) {
                            config.finders.push(answer.replace(/'/g, '\\\''));

                            program.prompt('One more? ', function(answer) {
                                isYes(answer) ? more() : more('!');
                            });
                        });
                    },
                    function() {
                        step();
                    });
                },
            ],
            function(err) {
                if (err !== null) {
                    print.error('Terminating');
                    process.exit(1);
                } else {
                    fs.existsSync(TANKER_DIR) || fs.mkdirSync(TANKER_DIR);
                    fs.writeFileSync(TANKER_CNF, tmpl(config));
                    print.info('Done. File has been saved in ' + TANKER_CNF);
                    process.exit(0);
                }
            });
        });
};
