let assert = require('assert');
let path = require('path');
let exists = require('fs').existsSync;
let _ = require('lodash');
let util = require('../lib/util');
let TANKER_DIR = '.tanker';
let TANKER_CNF = 'config.js';
let TANKER_RSL = 'resolver.js';
let TANKER_REF = 'ref.json';
let TANKER_MAP = 'map.json';
let TANKER_LOG = 'log.json';

module.exports = function(config) {
    let paths = config.paths;

    // Корень проекта
    paths.root || (paths.root = walkUnless(TANKER_DIR));
    assert(paths.root, 'Не удалось определить корень проекта');

    let inProjectRoot = path.resolve.bind(path, paths.root);

    // Директория .tanker
    paths.tankerDir || (paths.tankerDir = inProjectRoot(TANKER_DIR));
    assert(exists(paths.tankerDir), 'tankerDir указывает на несуществующую папку');

    let inTankerDir = path.resolve.bind(path, paths.tankerDir);

    // Проектный конфиг
    paths.tankerCnf || (paths.tankerCnf = inTankerDir(TANKER_CNF));
    assert(exists(paths.tankerCnf), 'Не найден проектный конфиг');

    // Проектный резолвер
    paths.tankerRsl || (paths.tankerRsl = inTankerDir(TANKER_RSL));

    // Файл с ревизией текущих переводов
    paths.tankerRef || (paths.tankerRef = inTankerDir(TANKER_REF));

    // Файл с картой ключей
    paths.tankerMap || (paths.tankerMap = inTankerDir(TANKER_MAP));

    // Файл с логом
    paths.tankerLog || (paths.tankerLog = inTankerDir(TANKER_LOG));

    // Ревизия текущих переводов
    config.tanker.hash || (config.tanker.hash = (util.saferequire(paths.tankerRef) || {}).sha1);

    // Фильтр-префикс для поиска файлов
    paths.fprefix = paths.fprefix ? inProjectRoot(paths.fprefix) : process.cwd();
    assert(paths.fprefix.indexOf(paths.root) === 0, 'fprefix не является поддиректорией проекта');

    // Флаг про возможность интерактивного взаимодействия
    config.interactive || (config.interactive = process.stdin.isTTY);

    config.logger = util.getLogger(config.paths.tankerLog);

    // Убираем служебные поля
    config.toJSON = function() {
        return _.omit(config, ['logger', 'interactive']);
    };

    // Передать персональный токен
    if (process.env.TANKER_API_TOKEN) {
        config.tanker.token = process.env.TANKER_API_TOKEN;
    }
};

/**
 * Функция проходит вверх по иерархии папок, начиная с cwd,
 * пока не найдет переданный "якорь" (название директории или файла).
 *
 * @param  {String} anchor
 * @returns {String|undefined} Абсолютный путь до папки, содержащей anchor
 */
function walkUnless(anchor) {
    let dirname = path.dirname;
    let home = process.env.HOME || process.env.HOMEPATH || process.env.USERPROFILE;
    let chain = process.cwd();
    let root;

    do {
        if (exists(path.join(chain, anchor))) {
            root = chain;
            break;
        }
        chain = dirname(chain);
    }
    while ([home, dirname(chain)].indexOf(chain) < 0);

    return root;
}
