/**
 * За основу взят ./json2bem.js
 *
 * Обработка ключей и переводов, собираем все переводы,
 * раскладываем по папочкам(*.bemhtml-i18n, *.bh-i18n, *.priv-i18n, *.js-i18n)
 * в зависимости от технологий в которых изпользовался перевод.
 **/

const fs = require('fs');
const path = require('path');
const assert = require('assert');
const _ = require('lodash');
const tanker = require('../..');
const tmpl = _.template('module.exports = ${ data };\n');

// Общие блоки, нужен для оптимизации
const COMMON = ['blocks', 'blocks-common', 'common.blocks'];

/**
 * Экспортируем объект в котором
 * ключ - путь файла
 * значение - строка, переводы в json формате
 *
 * @param {Object} arg Результать выполнения танкер-кита
 * @param {Object} arg.info Конфигурация с которой был запущен танкер-кит
 * @param {Object} arg.data Строка, ответ сервера
 * @returns {Object}
 */
let dispatcher = function(arg) {
    /*
     Формат ответа танкер сервера
     {
        ru: {
            'myKeyset': 'Перевод',
            'myKeyset2': 'Перевод-2',
             ...
        },
        en: {
            'myKeyset': 'Translation',
            'myKeyset2': 'Translation-2',
            ...
        },
        ...
     }
     */

    let config = arg.info.config;
    let tankerKeys = JSON.parse(arg.data);
    let langs = _.keys(tankerKeys);

    let result = {};

    return tanker
        .find(config)
        /**
         * Запускаем парсер
         * @param {Object} paths Результать выполнения файндеров
         */
        .then(function(paths) {
            return tanker.parse(paths, config);
        })
        /**
         * Запускаем парсер
         * @param {Object} parsedKeys Найденные переводы
         */
        .then(function(parsedKeys) {
            let levels;
            let commonLevels;
            let customLevels;

            /**
             * Удаляем из parsedKeys если в объекте(ключе) не
             * указан кейсет либо значение ключа, группируем по уровням переопределения
             * и получаем технологию
             */
            parsedKeys = _(parsedKeys).filter(function(key) {
                return key.keyset;
            })
                .forEach(setKeyInfo.bind(this, config))
                .groupBy('level')
                .value();

            // Уровни переопределения
            levels = _.keys(parsedKeys);

            // Общие уровни
            commonLevels = _.intersection(levels, COMMON);
            customLevels = _.xor(commonLevels, levels);

            /**
             * Мы получаем из танкера все подряд (свои и чужие ключи),
             * нам надо оставить в коде те ключи которые упомянуты у нас в коде.
             *
             * Бывает так что ключи не указаны в коде а приходять из бекэнда,
             * в таких случай мы добавляем в кейсет (:js, :bemhtml, :bh, :priv) и
             * таким образом форсируем загрузку переводов которых нет в коде
             */
            _(tankerKeys.ru).keys().forEach(function(keyset) {
                let orphanKeyset = _.first(commonLevels) || 'blocks-common';
                let orphanKeys = parsedKeys[orphanKeyset] || (parsedKeys[orphanKeyset] = []);
                let backendKey = /(.+):(js|bemhtml|bh|priv)$/g.exec(keyset);

                if (backendKey) {
                    _.forEach(tankerKeys.ru[keyset], function(item, key) {
                        orphanKeys.push({
                            key: key,
                            tech: backendKey[2],
                            keyset: backendKey[1],
                            level: orphanKeyset,
                            orphan: keyset,
                        });
                    });

                    parsedKeys[orphanKeyset] = orphanKeys;
                }
            });

            /**
             * Если перевод исползуется в общих и на кастомных уровнях
             * удаляем их из кастомных уровнях, чтобы уменьшить количество файлов
             */
            // @TODO: Надо оптимизировать
            _.forEach(commonLevels, function(commonLevel) {
                _.forEach(customLevels, function(customLevel) {
                    parsedKeys[customLevel] = _.filter(parsedKeys[customLevel], function(key) {
                        return _.isEmpty(_.where(parsedKeys[commonLevel], {
                            key: key.key,
                            tech: key.tech,
                            keyset: key.keyset,
                        }));
                    });
                });
            });

            /**
             * Черепикаем из переводы из ответа танкера
             */
            _.forEach(parsedKeys, function(keys) {
                _.forEach(keys, function(key) {
                    _.merge(result, cherryPick(key, tankerKeys, langs));
                });
            });

            // Удаляем кейсеты, в которых все ключи пустые,
            // тем самым не создавая файлы с пустыми переводами.
            // См. https://st.yandex-team.ru/SERP-53078
            Object.keys(result).forEach(function(filepath) {
                let isEmpty = !Object.keys(result[filepath]).some(function(keyset) {
                    return _.filter(result[filepath][keyset]).length;
                });

                if (isEmpty) {
                    delete result[filepath];
                }
            });

            /**
             * Формируем красивый json файл с переводами
             */
            return _.mapValues(result, function(data) {
                return setTplData(data, tmpl);
            });
        });
};

/**
 * Определяет уровень переопреденения и технологию.
 *
 * @param {Object} config
 * @param {Object} key
 */
function setKeyInfo(config, key) {
    let filePath = key.path;
    let customLevelResolver = config && config.bemDispatcher && config.bemDispatcher.levelResolver;
    let levelResolver = customLevelResolver || function(filePath) {
        return path.dirname(filePath).split('/').shift();
    };
    let tech = path.basename(filePath).split('.').slice(1).join('.');

    key.level = levelResolver(filePath);

    key.tech = {
        'priv.js': 'priv',
        'bh.js': 'bh',
        'bemhtml.js': 'bemhtml',
        'vanilla.js': 'vanilla',
    }[tech] || tech;
}

/**
 *
 * @param {Object} key Ключ
 * @param {Object} tankerKeys Данные из танкера
 * @param {Array} langs Доступные языки
 * @returns {Object}
 */
function cherryPick(key, tankerKeys, langs) {
    let result = {};
    let id = key.key;
    let keyset = key.keyset;
    let orphan = key.orphan;
    let keysetPath = getPathByKeyset(keyset, key.level, key.tech);

    _.forEach(langs, function(lang) {
        let filePath = path.join(keysetPath, '/' + lang + '.js');
        let data = {};

        data[keyset] = {};
        data[keyset][id] = objPick(tankerKeys, lang, orphan || keyset, id);

        result[filePath] = _.merge((result[filePath] || {}), data);
    });

    return result;
}
/**
 * Шаблонизация JSON объекта.
 *
 * @param {Object} tplData
 * @param {Function} tmpl
 * @returns {String}
 */
function setTplData(tplData, tmpl) {
    let sortedObj = sortObject(tplData);

    return tmpl({ data: JSON.stringify(sortedObj, function(key, value) {
        if (typeof value === 'string') value = value.trim(); // Удаляем висящие пробелы (FRONTEND-452).
        return value || undefined; // Удаляем ключи с пустыми строками (SERP-53078).
    }, 4) });
}

/**
 * Сортирует объект по ключу.
 *
 * @param {Object} obj
 * @returns {Object}
 */
function sortObject(obj) {
    let sortedObj = {};
    let keys = _.keys(obj);

    keys = _.sortBy(keys, function(key) {
        return key;
    });

    _.each(keys, function(key) {
        if (_.isPlainObject(obj[key])) {
            sortedObj[key] = sortObject(obj[key]);
        } else {
            sortedObj[key] = obj[key];
        }
    });

    return sortedObj;
}

/**
 * Позволяет извлекать значения вложенных полей объекта,
 * передавая путь до нужного поля в виде строки и
 * не проверяя наличие каждого промежуточного звена.
 *
 * @param {Object} obj
 * @returns {Object|undefined}
 */
function objPick(obj) {
    let chain = _.toArray(arguments).slice(1);

    chain.every(function(key) {
        return Boolean(obj = obj[key]);
    });

    return obj;
}

/**
 *
 * @param keyset
 * @param level
 * @param tech
 * @returns {String}
 */
function getPathByKeyset(keyset, level, tech) {
    let BEM_PARSER = /^([^_]+)(__[^_]+)?(?:(_[^_]+)(_[^_]+))?$/; // block__elem(_mod_val)?
    let block = (keyset.match(BEM_PARSER) || []).slice(1).map(function(item) { // [block, elem <, mod, val>]
        return item || '';
    });
    let elem = block[1];
    let noUnderscoreElemDir;

    assert(block.length, 'Некорректное имя кейсета (' + keyset + ')');

    if (elem) {
        noUnderscoreElemDir = path.normalize(
            [level]
                .concat(block[0])
                .concat(elem.replace(/^__/, ''))
                .join(path.sep)
        );

        fs.existsSync(noUnderscoreElemDir) && (block[1] = elem.replace(/^__/, ''));
    }

    return path.normalize(
        [level]
            .concat(block.slice(0, 3))
            .concat(keyset + '.' + (tech ? tech : 'priv') + '-i18n')
            .join(path.sep)
    );
}

dispatcher.setTplData = setTplData;
module.exports = dispatcher;
