let _ = require('lodash');
let sha = require('sha1');

/**
 * Рекурсивно обходит объект. Находит вложенные объекты, у которых
 * есть ключ/значение kv. Передает каждый такой объект в callback.
 * Для рекурсивного обхода используются возможности JSON.stringify.
 *
 * @param {Object} obj Объект
 * @param {Array} kv Два элемента ключ/значение
 * @param {Function} callback
 */
exports.walkAndInvoke = function(obj, kv, callback) {
    JSON.stringify(obj, function(key, val) {
        _.isObject(val) && (val[kv[0]] === kv[1]) && callback(val);

        return val;
    });
};

/**
 * Позволяет извлекать значения вложенных полей объекта,
 * передавая путь до нужного поля в виде строки и
 * не проверяя наличие каждого промежуточного звена.
 *
 * Звенья в пути разделяются точкой, варианты - вертикальной чертой.
 *
 * Пример:
 *      getNested({a:{b:{c:'value'}}}, 'a.b.c'); // 'value'
 *      getNested({a:'sorry, only a'}, 'a.b.c'); // undefined
 *
 * @param {Object} obj
 * @chain {String} chain Путь.
 */
exports.getNested = function(obj, chain) {
    chain.split('.').every(function(key) {
        key.split('|').some(function(subkey) {
            key = subkey;
            return obj[subkey] !== undefined;
        });

        return Boolean(obj = obj[key]);
    });

    return obj;
};

/**
 * Вычисляет sha-сумму фрагмента AST-дерева.
 * В формировании хеша не участвуют поля 'loc' и 'range'.
 *
 * @param {Object} node - Фрагмент AST
 * @chain {...String} [additions] - Эти строки будут участвовать в формировании хеша
 */
exports.getHash = function(node, additions) {
    additions = [].slice.call(arguments, 1).join('');

    return sha(additions + JSON.stringify(node, function(key, val) {
        return ~['loc', 'range'].indexOf(key) ? undefined : val;
    }));
};

/**
 * Парсит id формата [!]<id>[extra].
 *
 * [!] - Флаг про то, что не нужно загружать ключ из проекта в Танкер;
 * <id> - Обычная строка или сторокой формата [form1][form2][form3] для склоняемых ключей;
 * [extra] - Строка формата {#:комментарий, _:контекст}.
 *
 * Примеры:
 *     '![form1][form2][formN]{#:комментарий, _:контекст}'
 *     'ТЕСТ{_:не переводить}'
 *     '!long-long-text'
 *
 * @param {String} id
 * @chain {Object} Поля ключа.
 */
exports.getTFProps = function(id) {
    let parts = id.match(/^(!)?(.+?)(?:\{([^{]*)\})?$/);
    let isPlr = /^\[.+\]$/.test(parts[2]);
    let value = isPlr ? parts[2].slice(1, -1).split(/\] *\[/, 4) : parts[2];
    let extra = _.object((parts[3] || '').split(/, */).map(function(str) {
        return str.split(':', 2);
    }));

    return {
        upload: !parts[1],
        value: _.defaults(value, ['', '', '', '']),
        plural: isPlr,
        comment: extra['#'] || '',
        context: extra._ || '',
    };
};
