let _ = require('lodash');
let bemParser = require('./bem');
let isString = _.isString;
let PARAM_RE = /\{([^{\s]+)\}/g;

module.exports = function(arg) {
    return bemParser(arg).map(function(key) {
        let params = key.params || {};

        // Если ID ключа на английском:
        // - не выгружать в Танкер
        // - не считать, что ID как-то связано со значением ключа
        //
        // Если ID ключа не на английском:
        // - Сделать значение ключа равным ID
        if (isString(key.key) && /^[\w-]+$/.test(key.key)) {
            key.upload = false;
        } else {
            key.value = key.key;
        }

        // Если в параметрах ключа есть поля 'some', 'many' или 'none':
        // - Считать ключ склоняемым
        // - Считать первой формой склонения значение ключа, определенное выше
        // - Вторую и третью берем из параметров some и many соответственно
        // - В русском языке "четвертая форма" всегда равна третьей
        if (_.intersection(['some', 'many', 'none'], Object.keys(params)).length) {
            key.plural = true;
            key.value = [
                key.value,
                params.some || null,
                params.many || null,
                params.none || params.many || null,
            ];
        } else {
            key.plural = false;
        }

        key.comment = params.comment || '';
        key.context = params.context || '';

        // Переводим параметры в формат <i18n:param>param</i18n:param>
        key.value = makeCanonicalProps(key.value);

        // Если ключ склоняемый и содержит параметры:
        // - Переводим его в базовую форму
        // - Склонения выражаем через i18n:dynamic
        if (key.plural) {
            key.plural = false;
            key.value = makeDynamicKey(key.value);
        }

        return key;
    });
};

function makeDynamicKey(forms) {
    return [
        '<i18n:dynamic project="tanker" keyset="dynamic" key="plural_adv">',
        '<i18n:count><i18n:param>count</i18n:param></i18n:count>',
        '<i18n:one>' + (forms[0] || '') + '</i18n:one>',
        '<i18n:some>' + (forms[1] || '') + '</i18n:some>',
        '<i18n:many>' + (forms[2] || '') + '</i18n:many>',
        '<i18n:none>' + (forms[3] || '') + '</i18n:none>',
        '</i18n:dynamic>',
    ].join('');
}

function makeCanonicalProps(forms) {
    let result = [].concat(forms).map(function(form) {
        return !_.isString(form) ? form : form.replace(PARAM_RE, function(param, value) {
            return '<i18n:param>' + value + '</i18n:param>';
        });
    });

    return result.length > 1 ? result : result[0];
}
