let assert = require('assert');
let path = require('path');
let _ = require('lodash');
let vow = require('vow');
let vowfs = require('vow-fs');
let print = require('./util').print;
let keys = Object.keys;

/**
 * Извлекает ключи из файлов, прогоняя контент файлов через парсеры.
 *
 * Обрабатываются только файлы, для которых есть парсеры.
 * Загружаются только парсеры, для которых есть файлы.
 *
 * @param {String[]} paths Абсолютные пути до файлов
 * @param {Object}   config Конфиг
 *
 * @returns {Promise}  result
 * @returns {Object}   result.info
 * @returns {String[]} result.info.procPaths Файлы которые были обработаны парсерами
 * @returns {String[]} result.info.keysPaths Файлы в которых были найдены ключи
 * @returns {Object[]} result.data Ключи
 */
module.exports = function(paths, config) {
    config.logger('parse.startTime', new Date());

    assert(config.parsers, 'Нет информации о парсерах');

    let procPaths = [];
    let keysPaths = [];
    let logger = config.logger;
    let relative = path.relative.bind(path, config.paths.root);
    let pathsByExt = groupByExt(paths);
    let extensions = _.intersection(keys(config.parsers), keys(pathsByExt));
    let results = [];
    let defaults = {
        upload: null,
        hash: null,
        path: null,
        location: null,
        range: null,
        fragment: null,
        single: null,
        language: config.tanker.original,
        keyset: null,
        key: null,
        value: null,
        plural: null,
        comment: null,
        context: null,
        params: null,
    };

    extensions.forEach(function(ext) {
        let parser = require(config.parsers[ext]);

        pathsByExt[ext].forEach(function(path) {
            results.push(vowfs.read(path, 'utf8').then(function(content) {
                let relPath = relative(path);
                let parsing;

                try {
                    parsing = parser({ data: content, path: relPath, type: ext });
                } catch (e) {
                    parsing = [];
                    print.error('[' + e + '][Path: ' + relPath + ']');
                }

                procPaths.push(path);
                parsing.length && keysPaths.push(path);

                return parsing.map(function(key) {
                    return _.defaults(key, { path: relPath }, defaults);
                });
            }));
        });
    });

    return vow
        .all(results)
        .then(_.flatten)
        .then(function(keys) {
            logger('parse.procPaths', procPaths);
            logger('parse.keysPaths', keysPaths);
            logger('parse.finishTime', new Date());
            return {
                info: {
                    procPaths: procPaths,
                    keysPaths: keysPaths,
                },
                meta: {},
                data: keys,
            };
        });
};

/**
 * Принимает массив путей, возвращает хеш, в котором пути сгруппированы по расширениям.
 * Расширением считается все, что идет за последним слешом после первой точки.
 * Пути у которых нет расширения (к пр. пути до директорий) группируются по полю _trash.
 *
 * @param {String[]} paths Массив путей
 * @returns {Object} Хеш, в котором пути сгруппированы по расширениям
 */
function groupByExt(paths) {
    let basename = path.basename;
    let re = /(?:\.(.+))?$/; // Матчится любое имя + расширение (если есть).

    return paths.reduce(function(rslt, path) {
        let ext = basename(path).match(re)[1] || '_trash';

        (rslt[ext] || (rslt[ext] = [])).push(path);

        return rslt;
    }, {});
}
