let _ = require('lodash');
let vow = require('vow');
let api = require('../api');
let NoError = require('./util').NoError;

/**
 * Загружает ключи в Танкер.
 *
 * @param {Object[]} keys Ключи
 * @param {String}   data Данные для отправки
 * @param {Object}   config Конфиг
 *
 * @returns {Promise} result
 * @returns {Object}  result.meta
 * @returns {String}  result.meta.response Ответ Танкера
 * @returns {Boolean} result.data Флаг об успешности завершения операции
 */
module.exports = function(keys, data, config) {
    if (config.keysetAdapters.push !== undefined) {
        let processedData = config.keysetAdapters.push({ keys: keys, data: data });
        keys = processedData.keys;
        data = processedData.data;
    }

    config.logger('push.startTime', new Date());

    if (!keys.length) {
        throw new NoError('Нет новых ключей. Отправка данных не имеет смысла.');
    }

    return api
        .headKeysets(config.tanker)
        .then(function(keysets) {
            let prjKeysets = _(keys).pluck('keyset').uniq().value();
            let newKeysets = _.difference(prjKeysets, keysets);
            let addKeyset = api.addKeyset.bind(api, config.tanker);

            return vow.all(newKeysets.map(addKeyset));
        })
        .then(function() {
            return api.resource('/keysets/onlyadd/', config.tanker, {
                file: data,
                keyset: _.sample(keys).keyset,
            });
        })
        .then(function(response) {
            config.logger('push.finishTime', new Date());
            return {
                info: {},
                meta: {
                    response: response,
                },
                data: true,
            };
        });
};
