let _ = require('lodash');
let vow = require('vow');
let vowfs = require('vow-fs');
let async = require('async');
let program = require('commander');
let util = require('./util');
let print = util.print;
let isYes = util.isYes;
let doNothing = function() {};

require('colors');

module.exports = function(keys, config) {
    config.logger('resolve.startTime', new Date());

    let mapPath = config.paths.tankerMap;
    let map = util.saferequire(mapPath) || {};
    let settings = config.resolving;
    let resolving = getResolving(settings);

    keys = _.uniq(keys, 'hash');

    /**
     * Сортируем ключи на resolved (есть в карте) и unresolved (нет в карте);
     * Обновляем поля alwaysFresh у resolved ключей;
     * Прогоняем unresolved-ключи через проектный резолвер.
     */
    return vow.invoke(function() {
        let alwaysFresh = settings.alwaysFresh;
        let resolver = util.saferequire(config.paths.tankerRsl) || doNothing;
        let resolved = [];
        let unresolved = [];

        keys.forEach(function(key) {
            let mapKeys = map[key.hash];

            (mapKeys || []).forEach(function(mapKey) {
                _.extend(mapKey, _.pick(key, alwaysFresh));
            });

            mapKeys ? resolved.push(mapKeys) : unresolved.push(key);
        });

        return vow
            .all(unresolved.map(resolver))
            .then(function() {
                return [resolved, unresolved];
            });
    })
    /**
     * Формируем подсказки;
     */
        .spread(function(resolved, unresolved) {
            let hinters = {};
            let types = _.intersection(Object.keys(config.hinters), _.pluck(unresolved, 'type'));
            let hints = {};

            types.forEach(function(type) {
                hinters[type] = require(config.hinters[type]);
            });

            unresolved.forEach(function(key) {
                hints[key.hash] = !hinters[key.type] ? {} : hinters[key.type]({ data: key });
            });

            return vow.all([resolved, unresolved, hints]);
        })
    /**
     * Уточняем недостающие данные у пользователя:
     * - Общие (keyIssues);
     * - Специфичные (subIssues);
     * - Отдельно спрашиваем про склонения, если нужно.
     */
        .spread(function(resolved, unresolved, allHints) {
            let deferred = vow.defer();
            let total = unresolved.length;
            let index = 1;

            if (total === 0) {
                return resolved;
            }

            print.info(
                'Некоторые ключи требуют подтверждения.',
                'Всего ключей: ' + total + '.'
            );

            async.eachSeries(unresolved, function(key, nextKey) {
                let common = settings.common;
                let issues = resolving.getIssues(key);
                let keyIssues = _.intersection(issues, common);
                let subIssues = _.difference(issues, common);
                let hints = allHints[key.hash];

                resolving.print(key, index, total);

                async.waterfall([

                    resolveIssues.bind(null, key, keyIssues, hints),

                    function(nextKeyStep) {
                        let basis = _.clone(key);
                        let isOneKey = resolving.isOneKey(basis);
                        let subIndex = 1;

                        key = [];

                        isOneKey || print.list(
                            'Фрагмент кода соответствует нескольким ключам.',
                            'Нужно ввести информацию по каждому из них.'
                        );

                        async.forever(function(nextSubKey) {
                            let subKey = _.clone(basis);

                            isOneKey || resolving.print(subKey, subIndex, index, total);

                            async.waterfall([

                                resolveIssues.bind(null, subKey, subIssues, hints),

                                function(nextSubKeyStep) {
                                    resolving.isPluralKey(subKey) ? nextSubKeyStep(null) : nextSubKeyStep('!');
                                },

                                function(nextSubKeyStep) {
                                    let value = subKey.value = [].concat(subKey.value);
                                    let pointer = 0;

                                    console.log('Введите исходные значения для различных склонений.');

                                    async.eachSeries(['1', '2', '5', '0'], function(issue, nextIssue) {
                                        if (_.isString(value[pointer])) {
                                            console.log(issue + ': ' + value[pointer]);
                                            pointer++;
                                            nextIssue();
                                        } else if (subKey.language === 'ru' && issue === '0') {
                                            // В русском три склонения
                                            console.log(issue + ': ' + value[2]);
                                            value[pointer] = value[2];
                                            pointer++;
                                            nextIssue();
                                        } else {
                                            program.prompt(issue + ': ', function(answer) {
                                                value[pointer] = answer || '';
                                                pointer++;
                                                nextIssue();
                                            });
                                        }
                                    },
                                    function() {
                                        nextSubKeyStep(null);
                                    });
                                },
                            ],
                            function() {
                                key.push(subKey);

                                subIndex++;

                                isOneKey ? nextSubKey('!') : program.prompt('Еще один ключ? ', function(answer) {
                                    isYes(answer) ? nextSubKey() : nextSubKey('!');
                                });
                            });
                        },
                        function() {
                            nextKeyStep();
                        });
                    },
                ],
                function() {
                    resolved.push(key);
                    index++;
                    console.log('Готово');
                    nextKey();
                });
            },
            function(error) {
                if (error) {
                    print.error('Произошла ошибка. Данные не будут сохранены.');
                    deferred.reject(error);
                } else {
                    print.info(
                        'Все ключи успешно обработаны.',
                        'Данные будут сохранены в файл ' + mapPath
                    );
                    deferred.resolve(resolved);
                }
            });

            function resolveIssues(key, issues, hints, callback) {
                async.eachSeries(issues, function(issue, nextIssue) {
                    askAbout(issue, key[issue], hints[issue], function(answer) {
                        print.answer(issue, key[issue], answer);

                        key[issue] = answer;

                        if (resolving.isViableKey(key, issue)) {
                            nextIssue();
                        } else {
                            console.log('Уточнять остальные данные нет смысла');
                            callback('!');
                        }
                    });
                },
                function() {
                    callback();
                });
            }

            return deferred.promise();
        })
    /**
     * Обновляем карту ключей;
     * Удаляем из результата ключи, у которых хотя бы одно из витальных полей имеет false-значение.
     */
        .then(function(resolved) {
            resolved = _.flatten(resolved);

            return vowfs
                .write(mapPath, JSON.stringify(_.extend(map, _.groupBy(resolved, 'hash')), null, 4), 'utf8')
                .then(function() {
                    return resolved.filter(function(key) {
                        return resolving.isViableKey(key);
                    });
                });
        })
        .then(function(result) {
            config.logger('resolve.finishTime', new Date());
            return {
                info: {},
                meta: {},
                data: result,
            };
        });
};

function askAbout(issue, value, hints, callback) {
    let msgDict = {
        upload: 'Загружать в Танкер?',
        single: 'Этот фрагмент кода соответствует одному ключу?',
        language: 'Язык (двухбуквенный id)',
        keyset: 'Название кейсета',
        key: 'Название ключа',
        value: 'Исходное значение ключа',
        plural: 'Несклоняемый?!',
        comment: 'Комментарий для переводчиков',
        context: 'Контекст',
    };
    let prompt = '\n  : ';
    let message = msgDict[issue] || 'Укажите значение поля ' + issue;
    let list = ['Указать свой вариант', 'Оставить как есть (' + value + ')'].concat(hints || []);

    if (list.length > 2) {
        console.log(message);

        program.choose(list, function(i) {
            switch (i) {
                case 0:
                    program.prompt('Свой вариант: ', function(answer) {
                        callback(answer);
                    });
                    break;
                case 1:
                    callback(value);
                    break;
                default:
                    callback(list[i]);
            }
        });
    } else if (/\?!$/.test(message)) {
        program.prompt(message.slice(0, -1) + prompt, function(answer) {
            callback(!isYes(answer));
        });
    } else if (/\?$/.test(message)) {
        program.prompt(message + prompt, function(answer) {
            callback(isYes(answer));
        });
    } else {
        program.prompt(message + prompt, function(answer) {
            if (!answer && value !== null) {
                answer = value;
            }
            callback(answer);
        });
    }
}

function getResolving(s) {
    return {
        /**
         * Возвращает массив полей, требующих резолвинга.
         */
        getIssues: function(key) {
            let issues = Object.keys(key).reduce(function(issues, p) {
                if (~s.skip.indexOf(p)) { return issues }
                if (~s.neverTrust.indexOf(p)) { issues.push(p); return issues }
                if (~s.alwaysTrust.indexOf(p)) { return issues }
                if (key[p] === null) { issues.push(p); return issues }
                return issues;
            }, []);
            let ordered = _.intersection(s.order, issues);
            let other = _.difference(ordered, issues);

            return ordered.concat(other);
        },

        print: function(key) {
            let tick = '  ›';

            console.log('Ключ ' + [].slice.call(arguments, 1).join('/'));

            Object.keys(key).forEach(function(p) {
                let line = p + ': ' + key[p];
                let isTruth = key[p] !== null;
                let isVital = ~s.vital.indexOf(p);

                if (~s.skip.indexOf(p)) { return }
                if (~s.alwaysTrust.indexOf(p)) { isTruth = true }
                if (~s.neverTrust.indexOf(p)) { isTruth = false }

                console.log(
                    isTruth ? tick.green : tick.red,
                    isVital ? line.underline : line
                );
            });
        },

        isOneKey: function(key) {
            return s.integrity.every(function(p) { return key[p] });
        },

        isPluralKey: function(key) {
            return s.plurality.every(function(p) { return key[p] });
        },

        isViableKey: function(key, issue) {
            let result;

            if (issue) {
                result = (s.vital.indexOf(issue) < 0) || key[issue];
            } else {
                result = s.vital.every(function(p) { return key[p] });
            }

            return result;
        },
    };
}
