let path = require('path');
let write = require('fs').writeFileSync;
let _ = require('lodash');
let pd = require('pretty-data').pd;

require('colors');

// исключение для ситуаций, когда на самом деле ошибки нет
function NoError(message) {
    Error.call(this, message);

    this.name = 'NoError';
    this.message = message;

    if (Error.captureStackTrace) {
        Error.captureStackTrace(this, NoError);
    } else {
        this.stack = (new Error()).stack;
    }
}

NoError.prototype = Object.create(Error.prototype);

module.exports = {

    NoError: NoError,

    print: {

        opts: {
            method: 'log',
            tick: '• ',
            label: '[tanker] ',
            color: 'white',
        },

        get br() {
            console.log('');
            return this;
        },

        get info() {
            return this.print.bind(this, {
                method: 'info',
                color: 'green',
            });
        },

        get error() {
            return this.print.bind(this, {
                method: 'error',
                color: 'red',
            });
        },

        get question() {
            return this.print.bind(this, {
                color: 'blue',
            });
        },

        get list() {
            return this.print.bind(this, {
                tick: '',
                label: '',
            });
        },

        answer: function(issue, before, after) {
            this.print(
                { tick: '  › ', label: '' },
                issue + ': ' + before + ' → ' + after
            );
        },

        pretty: function() {
            let messages = _(arguments).toArray().flatten().value();

            console.log(messages.map(function(message) {
                if (/^\s*<\?xml/.test(message)) {
                    return pd.xml(message);
                }

                try {
                    return pd.json(message);
                } catch (e) {
                    return message;
                }
            }).join('\n'));
        },

        json: function() {
            let messages = _(arguments).toArray().flatten().value();

            console.log(messages.map(function(message) {
                return JSON.stringify(message, null, 4);
            }).join('\n'));
        },

        print: function(opts) {
            let message = _(arguments).toArray().flatten().value().slice(1);

            _.defaults(opts, this.opts);

            message.length && console[opts.method](message.map(function(line) {
                return opts.tick[opts.color] + opts.label + line;
            }).join('\n'));

            return this;
        },
    },

    saferequire: function() {
        try {
            return require(path.resolve.apply(path, arguments));
        } catch (e) {}
    },

    isYes: function(str) {
        return !~['no', 'n', '0', 'net', 'нет', 'т', 'тщ', 'ytn'].indexOf(str);
    },

    getLogger: function(logPath) {
        let log = {
            _: {
                startTime: new Date(),
            },
        };

        process.on('exit', function() {
            log._.finishTime = new Date();
            write(logPath, JSON.stringify(log, null, 4), 'utf8');
        });

        return function(key, val) {
            let chain = key.split('.');
            let parent = chain.slice(0, -1).reduce(function(result, key) {
                return result[key] || (result[key] = {});
            }, log);

            key = chain.pop();

            return val ? (parent[key] = val) : parent[key] || (parent[key] = {});
        };
    },
};
