const assert = require('assert');

const serialize = require('serialize-javascript');
const merge = require('lodash/merge');
const urlJoin = require('url-join');
const fetch = require('node-fetch');

const { staticPath } = require('./build/staticPaths');
const translations = require('./src/translations');
const createI18N = require('./src/i18n');
const { getSettings } = require('./src/settings');
const { textToJson, jsonToHtml } = require('./src/utils/render');
const { getDataParams, createFetch } = require('./src/utils/actions');
const fetchAction = createFetch(fetch);

const templates = require('./templates');

const instances = ['intranet', 'business'];

const jsSerializeOptions = { space: 0 };

if (process.env.DEBUG) {
    jsSerializeOptions.space = 2;
}

/**
 * @param {Object} params
 * @param {String} [params.lang='ru']
 * @param {String} [params.platform='desktop']
 * @param {String} [params.instance='intranet']
 * @returns {{ render, assets }}
 */
function create(params) {
    const settings = getSettings(params);
    const { lang, platform, instance } = settings;
    const customSettings = merge({}, params);

    assert(translations[lang], `lang field should be one of ${Object.keys(translations)} but got ${lang}`);
    assert(templates[platform], `platform should be one of ${Object.keys(templates)} but got ${platform}`);
    assert(instances.includes(instance), `instance should be one of ${instances} but got ${instance}`);

    const I18N = createI18N(translations[lang], lang);
    const BEMHTML = templates[platform];

    function bemjsonToHtml(content) {
        return jsonToHtml({
            block: 'context-provider',
            content,
            data: { I18N },
        }, BEMHTML);
    }

    async function getActionData(actionData, actionParams) {
        const { url, options } = getDataParams(actionData, actionParams, settings);

        return fetchAction(url, merge({
            headers: {
                // Почти для всех экшенов нужно передать заголовки
                // * X-Ya-Service-Ticket
                // * X-Ya-User-Ticket
                // Сейчас этого не сделано,
                // потому что реализованные сейчас SSR экшены не требуют авторизации/аутентификации.
                // Это надо будет сделать с первым экшеном, требующим их.
                // Нужно будет поддержать проверку тикетов на стороне wf-schi и
                // передачу заголовков в параметры на стороне wf-as-a-service
            },
        }, options));
    }

    function getConfig(params) {
        return merge({}, settings, params, { fetchAction: getActionData });
    }

    async function renderToString(source, params) {
        const json = await textToJson(source, getConfig(params));
        return bemjsonToHtml(json);
    }

    function getBrowserCode() {
        const cssUrl = urlJoin(staticPath, `${platform}.css`);
        const jsUrl = urlJoin(staticPath, `${platform}.js`);
        const jsConfigurationCode = `window.WIKI_FORMATTER_CONFIG = ${
            serialize(customSettings, jsSerializeOptions)
        };`;

        return {
            cssUrl,
            jsUrl,
            jsConfigurationCode,
        };
    }

    return {
        renderToString,
        getBrowserCode,

        // private stuff
        getConfig,
        bemjsonToHtml,
    };
}

exports.create = create;
