/* global io */
/* eslint-disable no-console */
const REQUEST_TIMEOUT = 180000;

const version = JSON.parse(process.env.VERSION);

const createIo = url => {
    const enqueued = [];
    const dequeued = new Set();

    let socket = null;

    const send = waiting => {
        const { action, params, accept, reject } = waiting;
        const requestTimeoutId = setTimeout(() => {
            dequeued.delete(waiting);
            reject({ slug: 'any', message: 'timeout' });
        }, REQUEST_TIMEOUT);

        dequeued.add(waiting);

        socket.emit(action, params, data => {
            if (!dequeued.has(waiting)) {
                // Значит ответ больше не требуется
                return;
            }

            clearTimeout(requestTimeoutId);
            dequeued.delete(waiting);

            if (data.error) {
                reject(data.error);
            } else {
                accept(data);
            }
        });
    };

    const runQueue = () => {
        while (socket.connected && enqueued.length > 0) {
            send(enqueued.shift());
        }
    };

    const initSocket = () => {
        // https://socket.io/docs/client-api
        socket = io(url, {
            // whether to reconnect automatically
            reconnection: true,
            // number of reconnection attempts before giving up
            reconnectionAttempts: Infinity,
            // how long to initially wait before attempting a new reconnection (1000).
            // Affected by +/- randomizationFactor,
            // for example the default initial delay will be between 500 to 1500ms.
            reconnectionDelay: 500,
            // maximum amount of time to wait between reconnections (5000).
            // Each attempt increases the reconnection delay
            // by 2x along with a randomization as above
            reconnectionDelayMax: 16000, // 500 * 2 * 2 * 2 * 2 * 2
            // 0 <= randomizationFactor <= 1
            randomizationFactor: 0, // Чтобы время реконнектов росло строго x2
            // by setting this false,
            // you have to call manager.open whenever you decide it’s appropriate
            autoConnect: true,
            // a list of transports to try (in order).
            // Engine always attempts to connect directly with the first one,
            // provided the feature detection test for it passes.
            transports: ['websocket'],
            // additional query parameters that are sent when connecting a namespace
            // (then found in socket.handshake.query object on the server-side)
            query: {
                _$client: `@yandex-int/woof@${version}`,
                _$origin: window.location.origin
            },
        });

        socket.on('connect', () => {
            console.info('Connect, success:', socket.id);
            runQueue();
        });

        socket.on('connect_error', error => {
            console.error('Connect, failure:', error);
        });

        socket.on('connect_timeout', timeout => {
            console.warn('Connect, timeout:', timeout);
        });

        socket.on('reconnect', attemptNumber => {
            console.info('Reconnect, success:', attemptNumber);
            runQueue();
        });

        socket.on('reconnect_attempt', attemptNumber => {
            console.info('Reconnect, attempt:', attemptNumber);
        });

        socket.on('disconnect', reason => {
            console.warn('Disconnect, reason:', reason);

            enqueued.push(...dequeued);
            dequeued.clear();

            if (reason === 'io server disconnect') {
                // the disconnection was initiated by the server, you need to reconnect manually
                socket.connect();
            }
            // else the manager will automatically try to reconnect
        });

        socket.on('error', error => {
            console.error('Uncaught socket.io error:', error);
        });
    };

    initSocket();

    document.addEventListener('visibilitychange', () => {
        if (document.hidden) {
            socket.close();
        } else {
            initSocket();
        }
    });

    return {
        get: ({ action, params }) => {
            return new Promise((accept, reject) => {
                enqueued.push({ action, params, accept, reject });

                runQueue();
            });
        },
    };
};

const WF_IO_CACHE = {};

const create = url => {
    if (!WF_IO_CACHE[url]) {
        WF_IO_CACHE[url] = createIo(url);
    }

    return WF_IO_CACHE[url];
};

exports.create = create;
