import { r as replaceTraps, j as instanceOfAny, k as reverseTransformCache, u as unwrap, d as deleteDatabase, a as openDBWithData } from './utils-66091b10.js';
import { d as dist_1, c as chai } from './index-a1a462bf.js';

const advanceMethodProps = ['continue', 'continuePrimaryKey', 'advance'];
const methodMap = {};
const advanceResults = new WeakMap();
const ittrProxiedCursorToOriginalProxy = new WeakMap();
const cursorIteratorTraps = {
    get(target, prop) {
        if (!advanceMethodProps.includes(prop))
            return target[prop];
        let cachedFunc = methodMap[prop];
        if (!cachedFunc) {
            cachedFunc = methodMap[prop] = function (...args) {
                advanceResults.set(this, ittrProxiedCursorToOriginalProxy.get(this)[prop](...args));
            };
        }
        return cachedFunc;
    },
};
async function* iterate(...args) {
    // tslint:disable-next-line:no-this-assignment
    let cursor = this;
    if (!(cursor instanceof IDBCursor)) {
        cursor = await cursor.openCursor(...args);
    }
    if (!cursor)
        return;
    cursor = cursor;
    const proxiedCursor = new Proxy(cursor, cursorIteratorTraps);
    ittrProxiedCursorToOriginalProxy.set(proxiedCursor, cursor);
    // Map this double-proxy back to the original, so other cursor methods work.
    reverseTransformCache.set(proxiedCursor, unwrap(cursor));
    while (cursor) {
        yield proxiedCursor;
        // If one of the advancing methods was not called, call continue().
        cursor = await (advanceResults.get(proxiedCursor) || cursor.continue());
        advanceResults.delete(proxiedCursor);
    }
}
function isIteratorProp(target, prop) {
    return ((prop === Symbol.asyncIterator &&
        instanceOfAny(target, [IDBIndex, IDBObjectStore, IDBCursor])) ||
        (prop === 'iterate' && instanceOfAny(target, [IDBIndex, IDBObjectStore])));
}
replaceTraps((oldTraps) => ({
    ...oldTraps,
    get(target, prop, receiver) {
        if (isIteratorProp(target, prop))
            return iterate;
        return oldTraps.get(target, prop, receiver);
    },
    has(target, prop) {
        return isIteratorProp(target, prop) || oldTraps.has(target, prop);
    },
}));

// Since this library proxies IDB, I haven't retested all of IDB. I've tried to cover parts of the
const { assert } = chai;
suite('Async iterators', () => {
    let db;
    teardown('Close DB', async () => {
        if (db)
            db.close();
        await deleteDatabase();
    });
    test('object stores', async () => {
        const schemaDB = await openDBWithData();
        db = schemaDB;
        {
            const store = schemaDB.transaction('key-val-store').store;
            const keys = [];
            const values = [];
            assert.isTrue(Symbol.asyncIterator in store);
            for await (const cursor of store) {
                dist_1(true);
                dist_1(true);
                dist_1(true);
                keys.push(cursor.key);
                values.push(cursor.value);
            }
            assert.deepEqual(values, [456, 123, 789], 'Correct values');
            assert.deepEqual(keys, ['bar', 'foo', 'hello'], 'Correct keys');
        }
        {
            const store = db.transaction('key-val-store').store;
            const keys = [];
            const values = [];
            for await (const cursor of store) {
                dist_1(true);
                dist_1(true);
                dist_1(true);
                keys.push(cursor.key);
                values.push(cursor.value);
            }
            assert.deepEqual(values, [456, 123, 789], 'Correct values');
            assert.deepEqual(keys, ['bar', 'foo', 'hello'], 'Correct keys');
        }
    });
    test('object stores iterate', async () => {
        const schemaDB = await openDBWithData();
        db = schemaDB;
        {
            const store = schemaDB.transaction('key-val-store').store;
            assert.property(store, 'iterate');
            dist_1(true);
            for await (const _ of store.iterate('blah')) {
                assert.fail('This should not be called');
            }
        }
        {
            const store = db.transaction('key-val-store').store;
            dist_1(true);
            for await (const _ of store.iterate('blah')) {
                assert.fail('This should not be called');
            }
        }
    });
    test('Can delete during iteration', async () => {
        const schemaDB = await openDBWithData();
        db = schemaDB;
        const tx = schemaDB.transaction('key-val-store', 'readwrite');
        for await (const cursor of tx.store) {
            cursor.delete();
        }
        assert.strictEqual(await schemaDB.count('key-val-store'), 0);
    });
    test('index', async () => {
        const schemaDB = await openDBWithData();
        db = schemaDB;
        {
            const index = schemaDB.transaction('object-store').store.index('date');
            const keys = [];
            const values = [];
            assert.isTrue(Symbol.asyncIterator in index);
            for await (const cursor of index) {
                dist_1(true);
                dist_1(true);
                dist_1(true);
                keys.push(cursor.key);
                values.push(cursor.value);
            }
            assert.deepEqual(values, [
                {
                    id: 4,
                    title: 'Article 4',
                    date: new Date('2019-01-01'),
                },
                {
                    id: 3,
                    title: 'Article 3',
                    date: new Date('2019-01-02'),
                },
                {
                    id: 2,
                    title: 'Article 2',
                    date: new Date('2019-01-03'),
                },
                {
                    id: 1,
                    title: 'Article 1',
                    date: new Date('2019-01-04'),
                },
            ], 'Correct values');
            assert.deepEqual(keys, [
                new Date('2019-01-01'),
                new Date('2019-01-02'),
                new Date('2019-01-03'),
                new Date('2019-01-04'),
            ], 'Correct keys');
        }
        {
            const index = db.transaction('object-store').store.index('title');
            const keys = [];
            const values = [];
            assert.isTrue(Symbol.asyncIterator in index);
            for await (const cursor of index) {
                dist_1(true);
                dist_1(true);
                dist_1(true);
                keys.push(cursor.key);
                values.push(cursor.value);
            }
            assert.deepEqual(values, [
                {
                    id: 1,
                    title: 'Article 1',
                    date: new Date('2019-01-04'),
                },
                {
                    id: 2,
                    title: 'Article 2',
                    date: new Date('2019-01-03'),
                },
                {
                    id: 3,
                    title: 'Article 3',
                    date: new Date('2019-01-02'),
                },
                {
                    id: 4,
                    title: 'Article 4',
                    date: new Date('2019-01-01'),
                },
            ], 'Correct values');
            assert.deepEqual(keys, ['Article 1', 'Article 2', 'Article 3', 'Article 4'], 'Correct keys');
        }
    });
    test('index iterate', async () => {
        const schemaDB = await openDBWithData();
        db = schemaDB;
        {
            const index = schemaDB.transaction('object-store').store.index('date');
            assert.property(index, 'iterate');
            dist_1(true);
            for await (const _ of index.iterate(new Date('2020-01-01'))) {
                assert.fail('This should not be called');
            }
        }
        {
            const index = db.transaction('object-store').store.index('title');
            assert.property(index, 'iterate');
            dist_1(true);
            for await (const _ of index.iterate('foo')) {
                assert.fail('This should not be called');
            }
        }
    });
    test('cursor', async () => {
        const schemaDB = await openDBWithData();
        db = schemaDB;
        const store = schemaDB.transaction('key-val-store').store;
        const cursor = await store.openCursor();
        if (!cursor)
            throw Error('expected cursor');
        const keys = [];
        const values = [];
        assert.isTrue(Symbol.asyncIterator in cursor);
        for await (const cursorIter of cursor) {
            dist_1(true);
            dist_1(true);
            dist_1(true);
            keys.push(cursorIter.key);
            values.push(cursorIter.value);
        }
        assert.deepEqual(values, [456, 123, 789], 'Correct values');
        assert.deepEqual(keys, ['bar', 'foo', 'hello'], 'Correct keys');
    });
});
