import { b as openDB, c as dbName, g as getNextVersion, w as wrap, u as unwrap, o as openDBWithSchema, d as deleteDatabase } from './utils-66091b10.js';
import { d as dist_1, c as chai } from './index-a1a462bf.js';

const { assert } = chai;
suite('openDb', () => {
    let db;
    teardown('Close DB', () => {
        if (db)
            db.close();
    });
    test('upgrade', async () => {
        let upgradeRun = false;
        const version = getNextVersion();
        db = (await openDB(dbName, version, {
            upgrade(db, oldVersion, newVersion, tx) {
                upgradeRun = true;
                dist_1(true);
                assert.instanceOf(db, IDBDatabase, 'db instance');
                assert.strictEqual(oldVersion, 0);
                assert.strictEqual(newVersion, version);
                dist_1(true);
                assert.instanceOf(tx, IDBTransaction, 'db instance');
                assert.strictEqual(tx.mode, 'versionchange', 'tx mode');
            },
        }));
        assert.isTrue(upgradeRun, 'upgrade run');
    });
    test('upgrade - schemaless', async () => {
        let upgradeRun = false;
        const version = getNextVersion();
        db = await openDB(dbName, version, {
            upgrade(db, oldVersion, newVersion, tx) {
                upgradeRun = true;
                dist_1(true);
                dist_1(true);
            },
        });
        assert.isTrue(upgradeRun, 'upgrade run');
    });
    test('blocked and blocking', async () => {
        let blockedCalled = false;
        let blockingCalled = false;
        let newDbBlockedCalled = false;
        let newDbBlockingCalled = false;
        db = (await openDB(dbName, getNextVersion(), {
            blocked() {
                blockedCalled = true;
            },
            blocking() {
                blockingCalled = true;
                // 'blocked' isn't called if older databases close once blocking fires.
                // Using set timeout so closing isn't immediate.
                setTimeout(() => db.close(), 0);
            },
        }));
        assert.isFalse(blockedCalled);
        assert.isFalse(blockingCalled);
        db = (await openDB(dbName, getNextVersion(), {
            blocked() {
                newDbBlockedCalled = true;
            },
            blocking() {
                newDbBlockingCalled = true;
            },
        }));
        assert.isFalse(blockedCalled);
        assert.isTrue(blockingCalled);
        assert.isTrue(newDbBlockedCalled);
        assert.isFalse(newDbBlockingCalled);
    });
    test('wrap', async () => {
        let wrappedRequest = Promise.resolve(undefined);
        // Let's do it the old fashioned way
        const idb = await new Promise(async (resolve) => {
            const request = indexedDB.open(dbName, getNextVersion());
            wrappedRequest = wrap(request);
            request.addEventListener('success', () => resolve(request.result));
        });
        assert.instanceOf(wrappedRequest, Promise, 'Wrapped request type');
        db = wrap(idb);
        dist_1(true);
        assert.instanceOf(db, IDBDatabase, 'DB type');
        assert.property(db, 'getAllFromIndex', 'DB looks wrapped');
        assert.strictEqual(db, await wrappedRequest, 'Wrapped request and wrapped db are same');
    });
    test('unwrap', async () => {
        const openPromise = openDB(dbName, getNextVersion());
        const request = unwrap(openPromise);
        dist_1(true);
        assert.instanceOf(request, IDBOpenDBRequest, 'Request type');
        db = (await openPromise);
        const idb = unwrap(db);
        dist_1(true);
        assert.instanceOf(idb, IDBDatabase, 'DB type');
        assert.isFalse('getAllFromIndex' in idb, 'DB looks unwrapped');
    });
});
suite('deleteDb', () => {
    let db;
    teardown('Close DB', () => {
        if (db)
            db.close();
    });
    test('deleteDb', async () => {
        db = (await openDBWithSchema());
        assert.lengthOf(db.objectStoreNames, 2, 'DB has two stores');
        db.close();
        await deleteDatabase();
        db = await openDB(dbName, getNextVersion());
        assert.lengthOf(db.objectStoreNames, 0, 'DB has no stores');
    });
    test('blocked', async () => {
        let blockedCalled = false;
        let blockingCalled = false;
        let closeDbBlockedCalled = false;
        db = await openDB(dbName, getNextVersion(), {
            blocked() {
                blockedCalled = true;
            },
            blocking() {
                blockingCalled = true;
                // 'blocked' isn't called if older databases close once blocking fires.
                // Using set timeout so closing isn't immediate.
                setTimeout(() => db.close(), 0);
            },
        });
        assert.isFalse(blockedCalled);
        assert.isFalse(blockingCalled);
        await deleteDatabase({
            blocked() {
                closeDbBlockedCalled = true;
            },
        });
        assert.isFalse(blockedCalled);
        assert.isTrue(blockingCalled);
        assert.isTrue(closeDbBlockedCalled);
    });
});
