const http = require('http');

/** @type {any} */
const got = require('got');

const MS_IN_SEC = 1000;
const SEC_IN_MIN = 60;
const MS_IN_MIN = SEC_IN_MIN * MS_IN_SEC;

const HTTP_STATUS_CODES = Object.keys(http.STATUS_CODES).map((code) => parseInt(code, 10));

const NON_RETRIABLE_STATUS_CODES = [
    404, // Not Found
    405, // Method Not Allowed
    413, // Payload To Large
    414, // URI Too Long
    418, // I'm a teapot
    431, // Request Header Fields Too Large
    508 // Loop Detected
];

/**
 * Максимальное время в миллисекундах между отправкой запроса и до получения ответа.
 */
const REQUEST_TIMEOUT_MS = MS_IN_MIN;

/**
 * Максимальное число повторных запросов.
 */
const MAX_RETRY_COUNT = 4;

/**
 * Методы, которые будут допущены к повторным запросам.
 *
 * @type {Set<string>}
 */
const RETRIABLE_METHODS = got.defaults.options.retry.methods;

/**
 * Статусы ответов, которые будут допущены к повторным запросам.
 */
const RETRIABLE_STATUS_CODES = HTTP_STATUS_CODES.filter((code) => {
    return code >= 400 && !NON_RETRIABLE_STATUS_CODES.includes(code);
});

/**
 * Коды сетевых ошибок, которые будут допущены к повторным запросам.
 *
 * Синхронизировано с `got`:
 * https://github.com/sindresorhus/got/blob/25f18be291dd866a37f5153ca76d44151fcd702b/source/is-retry-on-network-error-allowed.js#L4-L11
 */
const RETRIABLE_NETWORK_ERROR_CODES = [
    'ETIMEDOUT',
    'ECONNRESET',
    'EADDRINUSE',
    'ECONNREFUSED',
    'EPIPE',
    'ENOTFOUND',
    'ENETUNREACH',
    'EAI_AGAIN'
];

/**
 * Максимальная задержка между повторными запросами.
 */
const MAX_DELAY_BETWEEN_RETRIES_MS = 30 * MS_IN_MIN;

/**
 * Максимальное число символов, выводимое для тела неуспешного ответа.
 *
 * Ограничение необходимо для того, чтобы в логе не отображались огромные
 * HTML-ответы, например от GitHub-а.
 */
const TRUNCATE_RESPONSE_BODY_LOG_AFTER = 500;

module.exports = {
    HTTP_STATUS_CODES,
    REQUEST_TIMEOUT_MS,
    MAX_RETRY_COUNT,
    RETRIABLE_METHODS,
    RETRIABLE_STATUS_CODES,
    NON_RETRIABLE_STATUS_CODES,
    RETRIABLE_NETWORK_ERROR_CODES,
    MAX_DELAY_BETWEEN_RETRIES_MS,
    TRUNCATE_RESPONSE_BODY_LOG_AFTER
};
