const debug = require('debug')('si:ci:requests');

const utils = require('./utils');

/**
 * @param {Object} req
 * @param {Object} options
 * @returns {void}
 */
function logRequest(req, options) {
    const message = JSON.stringify({
        type: 'request',
        reqId: getRequestIdFromHeader(req),
        method: options.method,
        url: options.href
    });

    debug(message);
}

/**
 * @param {Object} res
 * @param {Object} options
 * @param {string} [body]
 * @returns {void}
 */
function logResponse(res, options, body) {
    if (utils.isResponseWithNetworkProblem(res) || utils.isTimedoutResponse(res)) {
        return logNetworkProblem(res);
    }

    const message = JSON.stringify({
        type: 'response',
        reqId: getRequestIdFromHeader(res.req || res),
        method: res.req ? res.req.method : res.method,
        url: res.url,
        statusCode: res.statusCode,
        statusMessage: res.statusMessage,
        /**
         * Логируем тело ответа только для неуспешных запросов.
         */
        body: res.statusCode >= 400 && body ? prepareResponseBody(body, options) : undefined
    });

    debug(message);
}

/**
 * @param {string} body
 * @param {Object} options
 * @returns {string|undefined}
 */
function prepareResponseBody(body, options) {
    if (body.length <= options.truncateResponseBodyLogAfter) {
        return body;
    }

    const truncatedBody = body.slice(0, options.truncateResponseBodyLogAfter);

    return `${truncatedBody}<truncated>`;
}

/**
 * @param {Object} res
 * @returns {void}
 */
function logNetworkProblem(res) {
    const message = JSON.stringify({
        type: 'response',
        reqId: getRequestIdFromHeader(res),
        method: res.method,
        url: res.url,
        networkCode: res.code
    });

    debug(message);
}

/**
 * @param {Object} res
 * @param {Object} options
 * @param {number} retryNumber
 * @param {number} nextRetryDelay
 * @returns {void}
 */
function logRetry(res, options, retryNumber, nextRetryDelay) {
    const nextRetryTime = Date.now() + nextRetryDelay;
    const nextRetryDate = new Date(nextRetryTime).toISOString();

    const message = JSON.stringify({
        type: 'retry-planned',
        reqId: getRequestIdFromHeader(res),
        method: res.method,
        url: res.url,
        retryNumber,
        maxRetryCount: options.retryCount,
        nextRetryDelay,
        maxRetryDelay: options.retry.maxRetryAfter,
        nextRetryDate
    });

    debug(message);
}

/**
 * @param {Object} res
 * @returns {void}
 */
function logSkippedRetryByNetworkError(res) {
    logSkippedRetry(res, `Skipped due to network error code: ${res.code}`);
}

/**
 * @param {Object} res
 * @returns {void}
 */
function logSkippedRetryByMethod(res) {
    logSkippedRetry(res, `Skipped due to non-retriable method: ${res.method}`);
}

/**
 * @param {Object} res
 * @returns {void}
 */
function logSkippedRetryByStatusCode(res) {
    logSkippedRetry(res, `Skipped due to non-retriable status code: ${res.statusCode}`);
}

/**
 * @param {Object} res
 * @param {Object} options
 * @returns {void}
 */
function logSkippedRetryByMaxRetryCount(res, options) {
    logSkippedRetry(res, `Skipped due to reached max retry count: ${options.retryCount}`);
}

/**
 * @param {Object} res
 * @param {Object} options
 * @param {number} nextRetryAfter
 * @returns {void}
 */
function logSkippedRetryByMaxRetryDelay(res, options, nextRetryAfter) {
    const nextRetryAfterInSecs = nextRetryAfter / 1000;
    const maxRetryAfterInSecs = options.retry.maxRetryAfter / 1000;

    logSkippedRetry(res, `Skipped due to 'maxRetryAfter' option: ${nextRetryAfterInSecs}s of ${maxRetryAfterInSecs}s`);
}

/**
 * @param {Object} res
 * @returns {void}
 */
function logSkippedRetryByMaxTotalRequestTime(res) {
    const expiresDate = new Date(res.headers['x-request-expires']).toISOString();

    logSkippedRetry(res, `Skipped due to reached max total request time. Expires request date: ${expiresDate}`);
}

/**
 * В силу того, что мы работаем и с ответами от сервера и с ошибками,
 * которые вызваны проблемами с сетью, приходится забирать заголовок по разному.
 *
 * В первом случае мы забираем заголовок из класса `ServerResponse`.
 * Во-втором случае из объекта.
 *
 * @param {Object|Response} res - ServerResponse или объект.
 * @returns {string}
 */
function getRequestIdFromHeader(res) {
    if (res.headers === undefined) {
        return res.getHeader('x-request-id');
    }

    return res.headers['x-request-id'];
}

/**
 * @param {Object} res
 * @param {string} reason
 * @returns {void}
 */
function logSkippedRetry(res, reason) {
    const message = JSON.stringify({
        type: 'retry-skipped',
        reqId: getRequestIdFromHeader(res),
        method: res.method,
        url: res.url,
        reason
    });

    debug(message);
}

module.exports = {
    logRequest,
    logResponse,
    logRetry,

    logSkippedRetryByNetworkError,
    logSkippedRetryByMethod,
    logSkippedRetryByStatusCode,
    logSkippedRetryByMaxRetryCount,
    logSkippedRetryByMaxRetryDelay,
    logSkippedRetryByMaxTotalRequestTime
};
