const crypto = require('crypto');

/** @type {any} */
const got = require('got');

const pkg = require('../package.json');
const logger = require('./logger');
const retries = require('./retries');
const {
    HTTP_STATUS_CODES,
    REQUEST_TIMEOUT_MS,
    RETRIABLE_METHODS,
    RETRIABLE_STATUS_CODES,
    NON_RETRIABLE_STATUS_CODES,
    MAX_DELAY_BETWEEN_RETRIES_MS,
    MAX_RETRY_COUNT,
    TRUNCATE_RESPONSE_BODY_LOG_AFTER
} = require('./constants');

const client = got.create({
    handler: (options, nextHandler) => {
        const requestId = getRequestId(options);
        const requestExpiresDate = getRequestExpiresDate(options);

        setRequestIdHeader(options, requestId);
        setRequestExpiresHeader(options, requestExpiresDate);
        setRetriableStatusCodes(options);

        /**
         * Подменяем функцию повторов запросов здесь потому, что нам нужно передать знания
         * между первым запросом и последующими повторными запросами.
         *
         * @param {number} retryNumber
         * @param {Object} error
         * @returns {number}
         */
        options.retry.retries = (retryNumber, error) => {
            setRequestIdHeader(error, requestId);
            setRequestExpiresHeader(error, requestExpiresDate);

            return retries.prepareRetry(retryNumber, error, options);
        };

        /**
         * Запрос может быть как потоком, так и обещанием.
         * Для каждого вида запроса необходим свой механизм обработки.
         */
        const req = nextHandler(options);

        req.on('request', (request) => logger.logRequest(request, options));
        req.on('response', (response) => {
            let body = '';

            const isLoggableResponseBody = response.statusCode >= 400;

            if (isLoggableResponseBody) {
                response.on('data', (chunk) => {
                    const isEmptyBody = body.length === 0;
                    const isTruncatedChunk = (body.length + chunk.length) <= options.truncateResponseBodyLogAfter;

                    if (isEmptyBody || !isTruncatedChunk) {
                        body += chunk;
                    }
                });
            }

            response.on('end', () => logger.logResponse(response, options, body));
        });

        /**
         * В случае ответа сервера ошибкой `got` не триггерит событие `response`, а сразу переходит к ошибке.
         */
        req.on('error', (error) => {
            setRequestIdHeader(error, requestId);

            logger.logResponse(error, options);
        });

        return req;
    },
    options: got.mergeOptions(got.defaults.options, {
        headers: {
            'user-agent': `@yandex-int/si.ci.requests@${pkg.version}`
        },
        timeout: REQUEST_TIMEOUT_MS,
        retry: {
            methods: RETRIABLE_METHODS,
            statusCodes: RETRIABLE_STATUS_CODES,
            maxRetryAfter: MAX_DELAY_BETWEEN_RETRIES_MS
        },
        nonRetriableStatusCodes: [],
        retryCount: MAX_RETRY_COUNT,
        applyMaxRetryTotalTimeBeforeRetry: true,
        truncateResponseBodyLogAfter: TRUNCATE_RESPONSE_BODY_LOG_AFTER
    })
});

module.exports = client;
module.exports.constants = {
    HTTP_STATUS_CODES,
    RETRIABLE_STATUS_CODES,
    NON_RETRIABLE_STATUS_CODES,
    RETRIABLE_METHODS
};
module.exports.withRetriablePostMethod = client.extend({
    retry: {
        methods: new Set(['POST', ...RETRIABLE_METHODS])
    }
});
module.exports.withRetriable404StatusCode = client.extend({
    retry: {
        statusCodes: new Set([404, ...RETRIABLE_STATUS_CODES])
    }
});

/**
 * Устанавливает ID-запроса в заголовок `x-request-id`.
 *
 * @param {Object} obj
 * @param {string} requestId
 * @returns {void}
 */
function setRequestIdHeader(obj, requestId) {
    obj.headers = obj.headers || {};
    obj.headers['x-request-id'] = requestId;
}

/**
 * @param {Object} options
 * @returns {string}
 */
function getRequestId(options) {
    if ('x-request-id' in options.headers) {
        return options.headers['x-request-id'];
    }

    return crypto.randomBytes(16).toString('hex');
}

/**
 * @param {Object} options
 * @returns {Common.HTTPDate|undefined}
 */
function getRequestExpiresDate(options) {
    if (options.maxRetryTotalTime === undefined) {
        return undefined;
    }

    const expiresTime = Date.now() + options.maxRetryTotalTime;

    return new Date(expiresTime).toUTCString();
}

/**
 * Устанавливает дату, по истечению которой ответ будет считаться устаревшим.
 *
 * Используем пользовательский заголовок `x-request-expires`, потому что ни один
 * заголовок из стандарта не подходит.
 *
 * @param {Object} obj
 * @param {Common.HTTPDate} [requestExpiresDate]
 * @returns {Object}
 */
function setRequestExpiresHeader(obj, requestExpiresDate) {
    if (requestExpiresDate === undefined) {
        return;
    }

    obj.headers = obj.headers || {};
    obj.headers['x-request-expires'] = requestExpiresDate;
}

/**
 * Устанавливает коды ответов, которые нужно ретраить.
 *
 * Предварительно клонирует текущий набор, чтобы не распространять изменения на другие клиенты.
 *
 * @param {Object} options
 * @returns {void}
 */
function setRetriableStatusCodes(options) {
    const statusCodes = new Set([...options.retry.statusCodes]);

    options.nonRetriableStatusCodes.forEach((code) => statusCodes.delete(code));

    options.retry.statusCodes = statusCodes;
}
