import crypta.adhoc.retargeting.config_pb2 as config
from crypta.lib.python.getoptpb import ParsePbOptions
from crypta.lib.python.yql.client import create_yql_client
from crypta.lib.python.logging import logging_helpers

import library.python.resource as rs
from yt.wrapper import YtClient, TablePath
from fbprophet import Prophet
import pandas as pd

import logging
import datetime


logger = logging.getLogger(__name__)


def parse_day(str_):
    return datetime.datetime.strptime(str(str_), '%Y-%m-%d').date()


def day_back(date, count):
    return date - datetime.timedelta(days=count)


def quote(str_):
    return '"%s"' % (str_)


def fit_predict(series, periods):
    model = Prophet(
        yearly_seasonality=False,
        weekly_seasonality=True,
        daily_seasonality=False,
        changepoint_range=0.9,
        changepoint_prior_scale=0.3,
        interval_width=0.75,
    )
    model.fit(series, iter=5000)
    future = model.make_future_dataframe(periods=periods)
    forecast = model.predict(future)
    df = forecast[["ds", "trend", "trend_lower", "trend_upper", "yhat", "yhat_lower", "yhat_upper"]]
    df = df.assign(
        dt=df.ds.dt.strftime("%Y-%m-%d"),
    ).drop(["ds"], axis=1)
    return df


def predict_and_store(yt, path, destination, value_column='Value', lookahead=180):
    series = pd.DataFrame(list(yt.read_table(path)))
    series = series[['Day', value_column]].rename(columns={'Day': 'ds', value_column: 'y'})
    schema = [
        {"name": "dt", "type": "string"},
        {"name": "trend", "type": "double"},
        {"name": "trend_lower", "type": "double"},
        {"name": "trend_upper", "type": "double"},
        {"name": "yhat", "type": "double"},
        {"name": "yhat_lower", "type": "double"},
        {"name": "yhat_upper", "type": "double"},
    ]
    yt.remove(destination, force=True)
    yt.create('table', destination, attributes={"schema": schema})
    predictions = fit_predict(series, lookahead)
    predictions.clip(lower=0, inplace=True)
    yt.write_table(destination, predictions.T.to_dict().values())


def get_target_days(args):
    origin = parse_day(args.Day) if args.Day else day_back(datetime.date.today(), 1)
    return set([str(day_back(origin, i)) for i in range(args.DaysBack)])


def get_computed_days(yt, tables, column='Day'):
    days = set()
    for table in tables:
        days |= set([row[column] for row in yt.read_table(table)])
    return days


def main():
    logging_helpers.configure_stderr_logger(logging.getLogger())
    args = ParsePbOptions(config.TConfig)
    root = TablePath(args.Paths.OutputDirectory)
    offer_retargeting_market_share = root.join("OfferRetargetingMarketShare")
    retargeting_share = root.join("RetargetingYandexShare")

    yql = create_yql_client(args.Yt.Proxy, args.Yt.Token, tmp_folder=args.Paths.TmpDirectory)
    yt = YtClient(proxy=args.Yt.Proxy, token=args.Yt.Token)

    target_days = get_target_days(args)
    computed_days = get_computed_days(yt, [offer_retargeting_market_share, retargeting_share])
    effective_days = target_days - computed_days
    logger.info('Going to compute metrics for [%s]', effective_days)

    query = rs.find('/query/metrics.yql').decode('utf-8').format(
        days=','.join(map(quote, effective_days)),
        offer_retargeting_market_share_metrics=offer_retargeting_market_share,
        retargeting_share_metrics=retargeting_share,
    )
    yql.execute(query, syntax_version=1)

    predict_and_store(yt, offer_retargeting_market_share, offer_retargeting_market_share + "Future")
    predict_and_store(yt, retargeting_share, retargeting_share + "Future")
    predict_and_store(yt, retargeting_share, root.join("RetargetingYandex") + "Future", value_column='RetargetingPrgg')
