from datetime import timedelta
import os

from crypta.affinitive_geo.services.org_embeddings.lib.utils import config
from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers


get_org_affinitive_banners_query = u'''
PRAGMA yt.UseNativeYtTypes;

$alpha = {{smoothing_alpha}};

$distinct_orgvisits = (
    SELECT
        CAST(orgvisits.IdValue AS Uint64) AS crypta_id,
        orgs_info.permalink AS permalink,
        orgs_info.region_id AS region_id,
    FROM `{{orgvisits_for_description_table}}` AS orgvisits
    INNER JOIN `{{orgs_info_table}}` AS orgs_info
    ON CAST(orgvisits.GroupID AS Uint64) == orgs_info.permalink
);

$shows_input = (
    SELECT
        {{shows_weight}} AS weight,
        crypta_id,
        groupbannerid,
        Geo::RoundRegionById(CAST(regionid AS Int32), 'region').id AS region_id,
    FROM `{{chevent_shows_table}}`
UNION ALL
    SELECT
        {{clicks_weight}} AS weight,
        crypta_id,
        groupbannerid,
        Geo::RoundRegionById(CAST(regionid AS Int32), 'region').id AS region_id,
    FROM `{{chevent_clicks_table}}`
);
$shows_input = (
    SELECT
        crypta_id,
        groupbannerid,
        region_id,
        MAX(weight) AS weight,
    FROM $shows_input
    GROUP BY crypta_id, groupbannerid, region_id
);

$showed_users = (
    SELECT
        region_id,
        CAST(COUNT(DISTINCT crypta_id) AS Double) AS users_cnt,
    FROM $shows_input
    GROUP BY region_id
);

$shows_by_banners = (
    SELECT
        groupbannerid,
        region_id,
        CAST(SUM(weight) AS Double) AS shows_cnt,
    FROM $shows_input
    GROUP BY groupbannerid, region_id
);

$banner_show_ratios = (
    SELECT
        shows.groupbannerid AS groupbannerid,
        shows.region_id AS region_id,
        shows.shows_cnt  / showed_users.users_cnt AS ratio,
    FROM $shows_by_banners AS shows
    INNER JOIN $showed_users AS showed_users
    USING(region_id)
);

$shows_with_orgs = (
    SELECT DISTINCT
        orgs.crypta_id AS crypta_id,
        orgs.permalink AS permalink,
        orgs.region_id AS region_id,
        shows.groupbannerid AS groupbannerid,
        shows.weight AS weight,
    FROM $distinct_orgvisits AS orgs
    INNER JOIN $shows_input AS shows
    USING(crypta_id)
);

$users_by_orgs = (
    SELECT
        permalink,
        region_id,
        CAST(COUNT(DISTINCT crypta_id) AS Double) AS shows_cnt,
    FROM $shows_with_orgs
    GROUP BY permalink, region_id
);

$shows_by_orgs_and_banners = (
    SELECT
        permalink,
        groupbannerid,
        region_id,
        CAST(SUM(weight) AS Double) AS shows_cnt,
    FROM $shows_with_orgs
    GROUP BY permalink, groupbannerid, region_id
);

$org_banner_show_ratios = (
    SELECT
        org_banner.permalink AS permalink,
        org_banner.region_id AS region_id,
        org_banner.groupbannerid AS groupbannerid,
        org_banner.shows_cnt AS org_banner_shows_cnt,
        org_banner.shows_cnt / org.shows_cnt  AS ratio,
    FROM $shows_by_orgs_and_banners AS org_banner
    INNER JOIN $users_by_orgs AS org
    USING(permalink, region_id)
);

$org_banner_affinities = (
    SELECT
        org_banner.permalink AS permalink,
        org_banner.groupbannerid AS groupbannerid,
        org_banner.region_id AS region_id,
        (org_banner.ratio / MAX_OF(banner.ratio, {{min_banner_ratio}}) * org_banner_shows_cnt + $alpha)
            / (org_banner_shows_cnt + $alpha) AS affinity,
    FROM $org_banner_show_ratios AS org_banner
    INNER JOIN $banner_show_ratios AS banner
    USING(groupbannerid, region_id)
);

$org_banner_affinities_ranked = (
    SELECT
        permalink,
        groupbannerid,
        affinity,
        ROW_NUMBER() OVER w AS banner_rank,
    FROM $org_banner_affinities
    WINDOW w AS (
        PARTITION BY permalink
        ORDER BY affinity DESC
    )
);

$active_banner_groups = (
    SELECT
        groupbannerid,
        SOME(banner_body) as banner_body,
    FROM `{{active_banners_table}}`
    GROUP BY groupbannerid
);

$org_banner_affinities_filtered = (
    SELECT
        banner_body.banner_body AS banner_body,
        org_banner.* WITHOUT org_banner.banner_rank,
    FROM $org_banner_affinities_ranked AS org_banner
    INNER JOIN $active_banner_groups AS banner_body
    USING(groupbannerid)
    WHERE org_banner.banner_rank <= {{banners_cnt_per_org}}
);

$org_banner_affinities_aggregated = (
    SELECT
        permalink,
        ToDict(AGGREGATE_LIST_DISTINCT(AsTuple(
            COALESCE(CAST(groupbannerid AS String), ''), banner_body))) AS top_banners,
    FROM $org_banner_affinities_filtered
    GROUP BY permalink
);

INSERT INTO `{{org_affinitive_banners_table}}`
WITH TRUNCATE

SELECT
    orgs_info.*,
    affinities.top_banners AS top_banners,
FROM $org_banner_affinities_aggregated AS affinities
INNER JOIN `{{orgs_info_table}}` AS orgs_info
USING(permalink)
ORDER BY permalink;
'''


def get(yt_client, yql_client, date):
    org_affinitive_banners_table = os.path.join(config.ORG_AFFINITIVE_BANNERS_DIR, date)

    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            templater.render_template(
                get_org_affinitive_banners_query,
                vars={
                    'smoothing_alpha': config.SMOOTHING_ALPHA,
                    'shows_weight': config.SHOWS_WEIGHT,
                    'clicks_weight': config.CLICKS_WEIGHT,
                    'min_banner_ratio': config.MIN_BANNER_RATIO,
                    'banners_cnt_per_org': config.BANNERS_CNT_PER_ORG,
                    'orgvisits_for_description_table': config.ORGVISITS_FOR_DESCRIPTION_TABLE,
                    'orgs_info_table': config.ORGS_INFO_TABLE,
                    'chevent_shows_table': config.CHEVENT_SHOWS_TABLE,
                    'chevent_clicks_table': config.CHEVENT_CLICKS_TABLE,
                    'active_banners_table': config.ACTIVE_BANNERS_TABLE,
                    'org_affinitive_banners_table': org_affinitive_banners_table,
                },
            ),
            title='YQL affinitive geo organizations affinitive banners',
            transaction=str(transaction.transaction_id),
        )

        yt_helpers.set_ttl(
            table=org_affinitive_banners_table,
            ttl_timedelta=timedelta(days=config.DAYS_TO_STORE_DAILY_TABLES),
            yt_client=yt_client,
        )
