"""
Crypta API client

Usage:
  crypta
  crypta lab sample create --path=<path> --idType=<idType> --idKey=<idKey> [--dateKey=<dateKey>] [--name=<name>] [--groupingKey=<groupingKey>] [--ttl=<ttl>] [--accessLevel=<accessLevel>]
  crypta lab sample update source <id> --path=<path>
  crypta lab sample create matching view <id> --hashingMethod=<hashingMethod> --identifier=<identifier> [--includeOriginal] [--key=<key>] [--scope=<scope>]
  crypta lab sample create crypta_id_statistics view <id>
  crypta lab sample create lookalike view <id> --outputCount=<outputCount> --useDates=<useDates>
  crypta lab sample get view <id> <view-id>
  crypta lab sample get views <id>
  crypta lab sample delete <id>
  crypta lab sample list
  crypta lab sample get <id>
  crypta graph get --id=<id> --idType=<idType> --matchingType=<matchingType> --matchingScope=<matchingScope>
  crypta iPython
  crypta (-h | --help)

Options:
  lab sample create                     https://api.crypta.yandex-team.ru/swagger/index.html#/lab/createSample
  lab sample update source              https://api.crypta.yandex-team.ru/swagger/index.html#/lab/updateSampleSource
  lab sample update state               https://api.crypta.yandex-team.ru/swagger/index.html#/lab/updateSampleState
  lab sample create matching view       https://api.crypta.yandex-team.ru/swagger/index.html#/lab/createMatchingView
  lab sample create crypta_id_statistics view         https://api.crypta.yandex-team.ru/swagger/index.html#/lab/createCryptaIdStatisticsView
  lab sample create lookalike view      https://api.crypta.yandex-team.ru/swagger/index.html#/lab/createLookalikeView
  lab sample get view                   https://api.crypta.yandex-team.ru/swagger/index.html#/lab/getSampleView
  lab sample get views                  https://api.crypta.yandex-team.ru/swagger/index.html#/lab/getSampleViews
  lab sample delete                     https://api.crypta.yandex-team.ru/swagger/index.html#/lab/deleteSample
  lab sample list                       https://api.crypta.yandex-team.ru/swagger/index.html#/lab/getSamples
  lab sample get                        https://api.crypta.yandex-team.ru/swagger/index.html#/lab/getSample
  graph get                             https://api.crypta.yandex-team.ru/swagger/index.html#/graph/getById

  <id>                                  Sample ID.
  <view-id>                             View ID.
  --path=<path>                         Path to the sample, e.g. //tmp/sample.
  --idType=<idType>                     Type of ID.
  --idKey=<idKey>                       Name of the field that contains ID.
  --dateKey=<dateKey>                   Name of the field that contains date.
  --name=<name>                         Name of the sample.
  --groupingKey=<groupingKey>           Name of the field to group by.
  --ttl=<ttl>                           TTL of tables (in seconds).
  --accessLevel=<accessLevel>           Access level that controls scope of the sample, ["PRIVATE", "SHARED", "PUBLIC"].
  --hashingMethod=<hashingMethod>       Method of identifiers hashing, ["HM_IDENTITY", "HM_MD5", "HM_SHA256"].
  --identifier=<identifier>             Target identifier, e.g. ["LAB_ID_YANDEXUID", "LAB_ID_IDFA_GAID", "LAB_ID_MM_DEVICE_ID", "LAB_ID_EMAIL", "LAB_ID_PHONE", "LAB_ID_PUID", "LAB_ID_LOGIN", "LAB_ID_UUID", "LAB_ID_CRYPTA_ID", "LAB_ID_DIRECT_CLIENT_ID"]
  --includeOriginal                     Whether to match with original identifiers.
  --key=<key>                           Name of the field that contains ID.
  --mode=<mode>                         Matching mode, [ "CRYPTA_ID_STATISTICS" ].
  --outputCount=<outputCount>           Number of uids to be found by create lookalike view method.
  --useDates=<useDates>                 Whether to use date associated with ids.
  --scope=<scope>                       Scope of matching, [ "CROSS_DEVICE", "IN_DEVICE" ].

"""  # noqa
from __future__ import print_function

import logging

import config
import json
import time
import sys
import library.python.oauth as lpo

from docopt import docopt
from crypta.lib.python.swagger import swagger, _to_proto
from crypta.lab.proto.sample_pb2 import Sample
from crypta.lab.proto.view_pb2 import TSampleView
from google.protobuf.json_format import MessageToJson, MessageToDict


class CLIClient(object):
    def __init__(self):
        self.arguments = docopt(__doc__)
        self.client = self.create_client()

    @staticmethod
    def create_client():
        client_id = 'c92ca88b9d664402be33802f57d76dd4'
        client_secret = 'bed57130101347d88ed097a7e7e90097'
        token = config.SWAGGER_TOKEN or lpo.get_token(client_id, client_secret)
        swagger_json_url = config.SWAGGER_JSON_URL
        client = swagger(swagger_json_url=swagger_json_url, token=token)
        return client

    @staticmethod
    def wait_for_ready(raw_view):
        logging.info("Waiting for result.")
        limit_time = time.time() + 1  # 3 hours to wait
        while time.time() < limit_time:
            print("Test")
            view = raw_view.result()
            if view.State == 'READY':
                return True
            time.sleep(5)
        return False

    def create_sample(
        self, path, id_type, id_key, date_key=None, name=None, grouping_key=None, ttl=None, access_level=None, max_groups_count=10
    ):
        sample = self.client.lab.createSample(path=path,
                                              idType=id_type,
                                              idKey=id_key,
                                              dateKey=date_key,
                                              name=name,
                                              groupingKey=grouping_key,
                                              ttl=ttl,
                                              accessLevel=access_level,
                                              maxGroupsCount=max_groups_count,
                                              ).result(),
        return MessageToJson(_to_proto(Sample, sample))

    def update_sample_source(self, path, id):
        self.client.lab.updateSampleSource(id=id, path=path).result()

    def update_sample_state(self, state, id):
        self.client.lab.updateSampleState(id=id, state=state).result()

    def create_matching_view(self, id, hashing_method, identifier, include_original, key=None, scope=None):
        view = self.client.lab.createMatchingView(id=id,
                                                  hashingMethod=hashing_method,
                                                  identifier=identifier,
                                                  includeOriginal=include_original,
                                                  key=key,
                                                  scope=scope).result()
        return MessageToJson(_to_proto(TSampleView, view))

    def create_cryptaid_statistics_view(self, id):
        cryptaid_statistics_view = self.client.lab.createCryptaIdStatisticsView(id=id).result()
        return MessageToJson(_to_proto(TSampleView, cryptaid_statistics_view))

    def create_lookalike_view(self, id, outputCount, useDates):
        lookalike_view = self.client.lab.createLookalikeView(id=id, outputCount=outputCount, useDates=useDates).result()
        return MessageToJson(_to_proto(TSampleView, lookalike_view))

    def get_sample_view(self, id, view_id):
        raw_view = self.client.lab.getSampleView(id=id, view_id=view_id)
        view = raw_view.result()
        if not CLIClient.wait_for_ready(raw_view):
            path = view.Path
            raise Exception("Time limit (3 hours) exceeded.\nView table path: {0}.".format(path))

        return MessageToJson(_to_proto(TSampleView, view))

    def get_sample_views(self, id):
        raw_views = self.client.lab.getSampleViews(id=id).result()
        views = [MessageToDict(_to_proto(TSampleView, view)) for view in raw_views]
        return json.dumps(views, indent=4)

    def delete_sample(self, id):
        self.client.lab.deleteSample(id=id).result()

    def get_sample_list(self):
        samples = self.client.lab.getSamples().result()
        sample_ids = [MessageToDict(_to_proto(Sample, sample)) for sample in samples]
        return json.dumps(sample_ids, indent=4)

    def get_sample(self, id):
        sample = self.client.lab.getSample(id=id).result()
        return MessageToJson(_to_proto(Sample, sample))

    def get_graph(self, id, id_type, matching_type, matching_scope):
        graph = self.client.graph.getById(id=id,
                                          idType=id_type,
                                          matchingType=matching_type,
                                          matchingScope=matching_scope).result()
        return json.dumps(graph, default=lambda o: o.__dict__, indent=4)

    def command_line_parsing(self):
        try:
            if self.arguments['lab']:
                if self.arguments['sample']:
                    if self.arguments['create'] and not self.arguments['view']:
                        path = self.arguments['--path']
                        name = self.arguments['--name']
                        id_type = self.arguments['--idType']
                        id_key = self.arguments['--idKey']
                        date_key = self.arguments['--dateKey']
                        grouping_key = self.arguments['--groupingKey']
                        ttl = self.arguments['--ttl']
                        access_level = self.arguments['--accessLevel']
                        sample_id = self.create_sample(path=path,
                                                       id_type=id_type,
                                                       id_key=id_key,
                                                       date_key=date_key,
                                                       name=name,
                                                       grouping_key=grouping_key,
                                                       ttl=ttl,
                                                       access_level=access_level)
                        print(sample_id)
                        exit(0)

                    if self.arguments['update'] and self.arguments['source']:
                        path = self.arguments['--path']
                        id = self.arguments['<id>']
                        self.update_sample_source(id=id, path=path)
                        exit(0)

                    if self.arguments['create'] and self.arguments['view'] and self.arguments['matching']:
                        id = self.arguments['<id>']
                        hashing_method = self.arguments['--hashingMethod']
                        identifier = self.arguments['--identifier']
                        include_original = self.arguments['--includeOriginal']
                        key = self.arguments['--key']
                        scope = self.arguments['--scope']
                        view = self.create_matching_view(id=id,
                                                         hashing_method=hashing_method,
                                                         identifier=identifier,
                                                         include_original=include_original,
                                                         key=key,
                                                         scope=scope)
                        print(view)
                        exit(0)

                    if self.arguments['create'] and self.arguments['view'] and self.arguments['crypta_id_statistics']:
                        id = self.arguments['<id>']
                        print(self.create_cryptaid_statistics_view(id=id))
                        exit(0)

                    if self.arguments['create'] and self.arguments['view'] and self.arguments['lookalike']:
                        id = self.arguments['<id>']
                        outputCount = self.arguments['--outputCount']
                        useDates = self.arguments['--useDates']
                        print(self.create_lookalike_view(id=id, outputCount=outputCount, useDates=useDates))
                        exit(0)

                    if self.arguments['get'] and self.arguments['view']:
                        id = self.arguments['<id>']
                        view_id = self.arguments['<view-id>']
                        view = self.get_sample_view(id=id, view_id=view_id)
                        print(view)
                        exit(0)

                    if self.arguments['get'] and self.arguments['views']:
                        id = self.arguments['<id>']
                        views = self.get_sample_views(id=id)
                        print(views)
                        exit(0)

                    if self.arguments['delete']:
                        id = self.arguments['<id>']
                        self.delete_sample(id=id)
                        exit(0)

                    if self.arguments['list']:
                        print(self.get_sample_list())
                        exit(0)

                    if self.arguments['get']:
                        id = self.arguments['<id>']
                        print(self.get_sample(id))
                        exit(0)
            elif self.arguments['graph']:
                if self.arguments['get']:
                    id = self.arguments['--id']
                    id_type = self.arguments['--idType']
                    matching_type = self.arguments['--matchingType']
                    matching_scope = self.arguments['--matchingScope']
                    print(self.get_graph(id=id, id_type=id_type, matching_type=matching_type,
                                         matching_scope=matching_scope))

            elif self.arguments['iPython']:
                import IPython
                IPython.embed()
            else:
                print(__doc__)
                exit(1)

        except Exception as ex:
            print(str(ex), file=sys.stderr)
            exit(1)


def main():
    client = CLIClient()
    client.command_line_parsing()
