package ru.yandex.crypta.api.config;

import java.util.Collections;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.LongFunction;

import org.quartz.Job;
import org.quartz.JobBuilder;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleScheduleBuilder;
import org.quartz.TriggerBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.api.proto.TApiServerConfig;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.job.DeleteOutdatedSamplesJob;
import ru.yandex.crypta.lab.job.DumpDatabaseJob;
import ru.yandex.crypta.lab.job.DumpSegmentsBuchhalterJob;
import ru.yandex.crypta.lab.proto.TLabSchedulerOptions;
import ru.yandex.crypta.lib.schedulers.Schedulers;
import ru.yandex.crypta.service.bmcategory.BMCategoryService;
import ru.yandex.crypta.service.dmp.DmpService;
import ru.yandex.crypta.service.entity.extractor.EntityExtractorService;
import ru.yandex.crypta.service.pages.PageService;

public class Scheduling {

    private final static Logger LOG = LoggerFactory.getLogger(Scheduling.class);

    private Scheduling() {
    }

    public static void setUp(TApiServerConfig config, ServiceProvider serviceProvider) {
        Schedulers schedulers = serviceProvider.get(Schedulers.class);
        setUpLocalScheduler(serviceProvider, schedulers);

        if (config.getDevelopment().getDisableScheduling()) {
            LOG.info("Scheduling configuration is disabled");
            return;
        }

        setUpQuartzScheduler(config, serviceProvider, schedulers);
    }

    private static void setUpLocalScheduler(ServiceProvider serviceProvider, Schedulers schedulers) {
        ScheduledExecutorService executor = schedulers.getExecutor();

        executor.execute(() -> serviceProvider.get(BMCategoryService.class).refreshCache());
        executor.execute(() -> serviceProvider.get(PageService.class).refreshAllPagesCache());
        executor.execute(() -> serviceProvider.get(DmpService.class).refreshMetaCache());
        executor.execute(() -> serviceProvider.get(LabService.class).initialize());
        executor.execute(() -> serviceProvider.get(EntityExtractorService.class).refresh());
    }

    public static <T extends Job> void scheduleSimpleJob(TApiServerConfig config, Scheduler quartz,
            Class<T> jobClass, SimpleScheduleBuilder schedule)
    {
        var jobId = jobClass.getCanonicalName();
        var jobDetail = JobBuilder
                .newJob()
                .withIdentity(jobId, config.getQuartz().getMainGroup())
                .ofType(jobClass)
                .build();
        var trigger = TriggerBuilder
                .newTrigger()
                .withIdentity(jobId, config.getQuartz().getMainGroup())
                .withSchedule(schedule)
                .startNow()
                .build();
        try {
            quartz.scheduleJob(jobDetail, Collections.singleton(trigger), true);
        } catch (SchedulerException e) {
            throw new RuntimeException(e);
        }
    }

    private static void setUpQuartzScheduler(TApiServerConfig config, ServiceProvider serviceProvider,
            Schedulers quartz)
    {
        Scheduler scheduler = quartz.getQuartz().get();
        TLabSchedulerOptions labOptions = serviceProvider.get(TLabSchedulerOptions.class);
        LongFunction<SimpleScheduleBuilder> repeat =
                (long period) -> SimpleScheduleBuilder.repeatMinutelyForever((int) period);

        scheduleSimpleJob(config, scheduler, DumpDatabaseJob.class,
                repeat.apply(labOptions.getDumpDatabaseMinutesPeriod()));
        scheduleSimpleJob(config, scheduler, DeleteOutdatedSamplesJob.class,
                repeat.apply(labOptions.getDeleteOutdatedSamplesMinutesPeriod()));
        scheduleSimpleJob(config, scheduler, DumpSegmentsBuchhalterJob.class,
                repeat.apply(labOptions.getDumpSegmentsBuchhalterMinutesPeriod()));
    }

}
