package ru.yandex.crypta.api.rest.resource.bs;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.validation.constraints.NotNull;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;

import NBSYeti.NCounter.CounterIds;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.glassfish.jersey.server.ParamException;

import ru.yandex.ads.quality.adv.machine.entity.extract.protos.TCategory;
import ru.yandex.crypta.clients.bigb.BigbClient;
import ru.yandex.crypta.clients.bigb.BigbIdType;
import ru.yandex.crypta.clients.bigb.CommonId;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.clients.yabs.YabsClient;
import ru.yandex.crypta.common.ws.jersey.IpAware;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lib.entity.counters.EntityCounters;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.service.entity.extractor.EntityExtractorService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

@Path("bs")
@Api(tags = {"bs"})
@Produces(JsonUtf8.MEDIA_TYPE)
@Singleton
public class BsResource extends IpAware {

    private static final TypeReference<Map<String, String>> MAP_TYPE_REFERENCE = new TypeReference<>() {
    };
    private final Cache<Integer, JsonNode> directDebugCookieCache = CacheBuilder.newBuilder()
            .expireAfterWrite(3, TimeUnit.MINUTES)
            .build();
    private final BigbClient bigb;
    private final YabsClient yabs;
    private final YtService yt;
    private final EntityExtractorService entityExtractor;

    private static class CounterKey {
        public long counterId;
        public long id;

        public CounterKey(long counterId, long id) {
            this.counterId = counterId;
            this.id = id;
        }

        @Override
        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (!(other instanceof CounterKey otherCounterKey)) {
                return false;
            }

            return this.counterId == otherCounterKey.counterId && this.id == otherCounterKey.id;
        }

        @Override
        public int hashCode() {
            return Objects.hash(counterId, id);
        }
    }

    @Inject
    public BsResource(YabsClient yabs, YtService yt, BigbClient bigb, EntityExtractorService entityExtractor) {
        this.bigb = bigb;
        this.yabs = yabs;
        this.yt = yt;
        this.entityExtractor = entityExtractor;
    }

    @GET
    @Path("bb")
    @ApiOperation(value = "Retrieves user profile")
    public Object getUserProfile(
            @QueryParam("uidType") @ApiParam(value = "BigB uid type") @NotNull BigbIdType bigbIdType,
            @QueryParam("uid") @ApiParam(value = "BigB uid value") @NotNull String bigbIdValue
    ) {
        return bigb.getBigbData(bigbIdType, bigbIdValue);
    }

    @GET
    @Path("select_type")
    @ApiOperation(value = "Retrieve select types")
    public List<SelectType> getSelectTypes() {
        List<SelectType> selectTypes = new ArrayList<>();
        YPath table = YPath.simple("//home/yabs/dict/SelectType");
        yt.getHahn().tables().selectRows(
                String.format("* FROM [%s] WHERE SelectType != 0", table),
                YTableEntryTypes.yson(SelectType.class),
                (Consumer<SelectType>) selectTypes::add
        );
        return selectTypes;
    }

    @GET
    @Path("enhanced_profile")
    @ApiOperation(value = "Retrieves pretty user profile")
    @RolesAllowed({Roles.Portal.PROFILE})
    public Object getEnhancedUserProfile(
            @QueryParam("uidType") @ApiParam(value = "BigB uid type") BigbIdType uidType,
            @QueryParam("uidValue") @ApiParam(value = "BigB uid value") String uidValue,
            @QueryParam("matching") @ApiParam(value = "Use matching") int matching,
            @Context HttpHeaders httpHeaders
    ) {
        return bigb.getEnhancedProfile(uidType, uidValue, matching, httpHeaders.getHeaderString("Cookie"));
    }

    @GET
    @Path("search_text")
    @ApiOperation(value = "Retrieves user search_text value from enhanced bb profile")
    @RolesAllowed({Roles.Portal.PROFILE})
    public Object getUserSearchText(
            @QueryParam("uidType") @ApiParam(value = "BigB uid type") BigbIdType uidType,
            @QueryParam("uidValue") @ApiParam(value = "BigB uid value") String uidValue,
            @QueryParam("matching") @ApiParam(value = "Use matching") int matching,
            @Context HttpHeaders httpHeaders
    ) {
        return bigb.getUserSearchText(uidType, uidValue, matching, httpHeaders.getHeaderString("Cookie"));
    }

    @GET
    @Path("search_text_with_matching")
    @ApiOperation(value = "Retrieve user search_text including profile with matching")
    @RolesAllowed({Roles.Portal.PROFILE})
    public Object getUserSearchTextWithMatching(
            @QueryParam("uidType") @ApiParam(value = "BigB uid type") BigbIdType uidType,
            @QueryParam("uidValue") @ApiParam(value = "BigB uid value") String uidValue,
            @Context HttpHeaders httpHeaders
    ) {
        return bigb.getUserSearchTextWithMatching(uidType, uidValue, httpHeaders.getHeaderString("Cookie"));
    }

    private Map<String, String> parseQueryParams(String queryString) {
        try {
            return new ObjectMapper().readValue(queryString, MAP_TYPE_REFERENCE);
        } catch (IOException e) {
            throw new ParamException.QueryParamException(e, "queryString", queryString);
        }
    }

    @GET
    @Path("ad")
    @ApiOperation(value = "Gets ad from BK")
    @RolesAllowed({Roles.Portal.ADS})
    @Deprecated
    public Object getAd(
            @QueryParam("pageId") @ApiParam(value = "Page Id") @NotNull String pageId,
            @QueryParam("queryParams") @ApiParam(value = "Ad query params") String queryString,
            @QueryParam("userId")
            @ApiParam(value = "User id in format uid_type=uid_value; supported types: yandexuid, uuid")
                    String userId,
            @QueryParam("userAgent") String userAgent,
            @QueryParam("yabsExpSid") String yabsExpSid,
            @QueryParam("debugMode") YabsClient.DebugMode debugMode
    ) {
        Map<String, String> queryParams = parseQueryParams(queryString);

        return yabs.getAd(pageId, queryParams, userId, userAgent, yabsExpSid, debugMode, getRemoteAddr());
    }

    @GET
    @Path("ad_json")
    @ApiOperation(value = "Gets ad from BK in JSON format")
    @RolesAllowed({Roles.Portal.ADS})
    public Object getAdJson(
            @QueryParam("pageId") @ApiParam(value = "Page Id") @NotNull String pageId,
            @QueryParam("queryParams") @ApiParam(value = "Ad query params") String queryString,
            @QueryParam("userId")
            @ApiParam(value = "User id in format uid_type=uid_value; supported types: yandexuid, uuid")
                    String userId,
            @QueryParam("userAgent") String userAgent,
            @QueryParam("yabsExpSid") String yabsExpSid,
            @QueryParam("debugMode") YabsClient.DebugMode debugMode
    ) {
        Map<String, String> queryParams = parseQueryParams(queryString);

        return yabs.getAdJson(pageId, queryParams, userId, userAgent, yabsExpSid, debugMode, getRemoteAddr());
    }

    @GET
    @Path("ad_debug")
    @ApiOperation(value = "Gets ad with full debug info")
    public Object getAdDebug(
            @QueryParam("pageId") @ApiParam(value = "Page Id") @NotNull String pageId,
            @QueryParam("queryParams") @ApiParam(value = "Ad query params") String queryString,
            @QueryParam("userId")
            @ApiParam(value = "User id in format uid_type=uid_value; supported types: yandexuid, uuid")
                    String userId,
            @QueryParam("userAgent") String userAgent,
            @QueryParam("yabsExpSid") String yabsExpSid
    ) {
        Map<String, String> queryParams = parseQueryParams(queryString);

        return yabs.getAdDebug(pageId, queryParams, userId, userAgent, yabsExpSid, getRemoteAddr());
    }

    @GET
    @Path("entity_counters")
    @ApiOperation(value = "Retrieves pretty user entity conuters")
    @RolesAllowed({Roles.Portal.PROFILE})
    public ru.yandex.crypta.api.proto.EntityCounters getEntityCounters(
            @QueryParam("uidType") @ApiParam(value = "BigB uid type") BigbIdType uidType,
            @QueryParam("uidValue") @ApiParam(value = "BigB uid value") String uidValue,
            @Context HttpHeaders httpHeaders
    ) {
        var counterPacks = bigb.getCounters(uidType, uidValue, EntityCounters.getAllCounterIds());
        var entityMapping = entityExtractor.getNamesCached();

        var builder = ru.yandex.crypta.api.proto.EntityCounters.newBuilder();
        var counterBuilders = new HashMap<CounterKey, ru.yandex.crypta.api.proto.EntityCounters.EntityCounter.Builder>();

        var counterGroupBuilders = new HashMap<Integer, ru.yandex.crypta.api.proto.EntityCounters.EntityCounterGroup.Builder>();
        var briefCounterBuilders = new HashMap<CounterKey, ru.yandex.crypta.api.proto.EntityCounters.EntityCounterBrief.Builder>();

        var ts = System.currentTimeMillis() / 1000L;

        for (var counterPack : counterPacks) {
            for (int i = 0; i < counterPack.getCounterIdsCount(); ++i) {
                var counterId = (int) counterPack.getCounterIds(i);
                var counterInfo = EntityCounters.getCounterInfo(counterId);
                var counterNameMap = entityMapping.get(counterInfo.type);

                var counterGroupBuilder = counterGroupBuilders.computeIfAbsent(
                        counterInfo.valueCounterId,
                        (key) -> builder.addEntityCounterGroupsBuilder()
                                .setType(TCategory.EType.valueOf(counterInfo.type))
                                .setCounterId(CounterIds.ECounterId.forNumber(counterInfo.valueCounterId))
                                .setDescription(counterInfo.description)
                );

                for (int j = 0; j < counterPack.getKeysCount(); ++j) {
                    var id = counterPack.getKeys(j);
                    var counterKey = new CounterKey(counterInfo.valueCounterId, id);
                    var counterBuilder = counterBuilders.computeIfAbsent(
                            counterKey,
                            (key) -> builder.addEntityCountersBuilder()
                                    .setType(TCategory.EType.valueOf(counterInfo.type))
                                    .setId(Long.toUnsignedString(id))
                                    .setName(counterNameMap.get(id))
                                    .setCounterId(CounterIds.ECounterId.forNumber(counterInfo.valueCounterId))
                    );
                    var briefCounterBuilder = briefCounterBuilders.computeIfAbsent(
                            counterKey,
                            (key) -> counterGroupBuilder.addCountersBuilder()
                                    .setId(Long.toUnsignedString(id))
                                    .setName(counterNameMap.get(id))
                    );
                    if (counterId == counterInfo.valueCounterId) {
                        var value = counterPack.getValues(i).getFloatValues().getValue(j);
                        counterBuilder.setValue(value);
                        briefCounterBuilder.setValue(value);
                    } else {
                        var ageSec = ts - counterPack.getValues(i).getFixed32Values().getValue(j);
                        counterBuilder.setAgeSec(ageSec);
                        briefCounterBuilder.setAgeSec(ageSec);
                    }
                }
            }
        }
        return builder.build();
    }

    @GET
    @Path("update_debug_cookie")
    @ApiOperation(value = "Gets Direct debug cookie")
    @RolesAllowed({Roles.Portal.ADS, Roles.Portal.PROFILE})
    public JsonNode updateDebugCookie() {
        return Caching.fetch(directDebugCookieCache, 1, this::fetchDebugCookie);
    }

    private JsonNode fetchDebugCookie() {
        return yabs.updateDebugCookie();
    }

    @GET
    @Path("bb/common_profiles/{common_id}")
    @ApiOperation(value = "Retrieves common profile")
    public Object getCommonProfile(
            @PathParam("common_id") @ApiParam(value = "Common profile id") CommonId commonId
    ) {
        return bigb.getCommonProfile(commonId);
    }

}
