package ru.yandex.crypta.api.rest.resource.ext;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.glassfish.jersey.server.ParamException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.api.rest.resource.ext.cdp.CdpResource;
import ru.yandex.crypta.api.rest.resource.ext.direct.DirectCaResource;
import ru.yandex.crypta.api.rest.resource.ext.lilucrm.LiluCrmResource;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.Yandexuid;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.service.bmcategory.BMCategoryService;
import ru.yandex.crypta.service.domains.DomainService;
import ru.yandex.crypta.service.pages.PageService;
import ru.yandex.crypta.service.useragents.UseragentsService;
import ru.yandex.inside.yt.kosher.cypress.RangeLimit;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.number.UnsignedLong;

@Path("ext")
@Api(tags = {"ext"})
@Produces(JsonUtf8.MEDIA_TYPE)
@Singleton
public class ExtResource {

    // TODO parametrize
    private static final String TOP_SITES_DICT = "//home/crypta/production/profiles/export/top_site_dict";
    private static final String BB_STORAGE = "//home/crypta/production/profiles/export/profiles_for_14days";
    private static final TypeReference<List<String>> LIST_TYPE_REFERENCE = new TypeReference<List<String>>() {
    };
    private final YtService ytService;
    private final UseragentsService useragentsService;
    private final BMCategoryService bmCategoryService;
    private final PageService pageService;
    private final DomainService domainService;
    private final Cache<Integer, Map<String, String>> topSitesCache = CacheBuilder.newBuilder()
            .expireAfterWrite(1, TimeUnit.DAYS)
            .build();
    private final Cache<Yandexuid, Map<String, Double>> topAffinitySitesCache = CacheBuilder.newBuilder()
            .expireAfterWrite(4, TimeUnit.HOURS)
            .build();
    private final Cache<String, List<Long>> pageIdsCache = CacheBuilder.newBuilder()
            .maximumSize(100)
            .build();

    @Inject
    public ExtResource(
            YtService ytService, UseragentsService useragentsService,
            BMCategoryService bmCategoryService, PageService pageService,
            DomainService domainService
    )
    {
        this.ytService = ytService;
        this.useragentsService = useragentsService;
        this.bmCategoryService = bmCategoryService;
        this.pageService = pageService;
        this.domainService = domainService;
    }


    @GET
    @Path("top_sites")
    public Map<String, String> getTopSites() {
        return Caching.fetch(topSitesCache, 0, this::retrieveTopSites);
    }

    private Map<String, String> retrieveTopSites() {
        ListF<YTreeMapNode> result = Cf.arrayList();
        ytService.getHahn().tables().read(YPath.simple(TOP_SITES_DICT),
                YTableEntryTypes.YSON, (Consumer<YTreeMapNode>) result::add);
        return result.stream().collect(Collectors.toMap(
                each -> Long.toUnsignedString(each.getLong("site_id")),
                each -> each.getString("site")
        ));
    }

    @GET
    @Path("top_affinity_sites")
    @ApiOperation(value = "Gets user top affinity sites")
    public Map<String, Double> getTopAffinitySites(
            @QueryParam("yandexuid") Yandexuid yandexuid
    )
    {
        return Caching.fetch(topAffinitySitesCache, yandexuid, () -> this.retrieveTopAffinitySites(yandexuid));
    }

    private Map<String, Double> retrieveTopAffinitySites(Yandexuid yandexuid) {
        ListF<YTreeMapNode> rows = Cf.arrayList();
        UnsignedLong yandexuidUL = UnsignedLong.valueOf(Long.parseUnsignedLong(yandexuid.getValue()));

        ytService.getHahn()
                .tables()
                .read(YPath.simple(BB_STORAGE).withExact(
                        new RangeLimit(Cf.list(YTree.builder().value(yandexuidUL).build()), -1, -1)),
                        YTableEntryTypes.YSON,
                        (Consumer<YTreeMapNode>) rows::add);

        ListF<YTreeNode> affinitiveSites = Option.wrap(rows.firstO()
                .getOrThrow(Exceptions::notFound)
                .get("affinitive_sites"));

        return asMap(affinitiveSites).mapValues(YTreeNode::doubleValue);
    }

    private MapF<String, YTreeNode> asMap(ListF<YTreeNode> affinitiveSites) {
        try {
            return Cf.wrap(affinitiveSites.get(0).asMap());
        } catch (UnsupportedOperationException e) {
            return Cf.map();
        }
    }

    @GET
    @Path("urls_page_ids")
    @RolesAllowed({Roles.Portal.ADS})
    public Map<String, List<Long>> getUrlsPageIds() {
        return Caching.fetchLoading(pageService.getAllPagesCache(), 0).siteNameToIds;
    }

    private List<Long> extractPageIdsByUrl(String url) {
        Map<String, List<Long>> pageIdsList = getUrlsPageIds();
        List<Long> pages = pageIdsList.get(url);

        if (pages == null) {
            throw Exceptions.notFound();
        }

        return pages;
    }

    @GET
    @Path("page_ids")
    @ApiOperation(value = "Get page ids of specified url")
    @RolesAllowed({Roles.Portal.ADS})
    public List<Long> getPageIdsByUrl(
            @QueryParam("url") @ApiParam(value = "Site url") String url)
    {
        return Caching.fetch(pageIdsCache, url, () -> extractPageIdsByUrl(url));
    }

    @GET
    @Path("get_useragents")
    @ApiOperation(value = "Get top useragents")
    public Map<String, List<String>> getTopUserAgents() {
        return useragentsService.getTopUseragents();
    }

    @PUT
    @Path("update_useragents")
    public void updateTopUseragent() {
        useragentsService.updateTopUseragents();
    }

    private List<String> parseIdList(String idList) {
        try {
            return new ObjectMapper().readValue(idList, LIST_TYPE_REFERENCE);
        } catch (IOException e) {
            throw new ParamException.QueryParamException(e, "idList", idList);
        }
    }

    @GET
    @Path("bm_categories")
    @ApiOperation(value = "Gets BMCategories description by key")
    @RolesAllowed({Roles.Portal.ADS})
    public Map<String, String> getBMCategoriesByIds(
            @QueryParam("ids") @ApiParam(value = "List of category ids") String ids
    )
    {
        List<String> idList = parseIdList(ids);
        var categories = Caching.fetchLoading(bmCategoryService.getCache(), 0);

        return idList.stream().collect(Collectors.toMap(
                categoryId -> categoryId,
                categoryId -> categories.getOrDefault(Long.valueOf(categoryId), "")
        ));
    }

    @GET
    @Path("target_domains")
    @ApiOperation(value = "Gets target domain names by ids")
    @RolesAllowed({Roles.Portal.ADS})
    public Map<String, String> getUserTargetDomainsByIds(
            @QueryParam("yandexuid") @ApiParam(value = "user Yandeuid") Yandexuid yandexuid,
            @QueryParam("ids") @ApiParam(value = "List of domain ids") String stringedIds
    )
    {
        List<Integer> ids = new ArrayList<>();
        parseIdList(stringedIds).forEach(id -> ids.add(Integer.valueOf(id)));

        Map<Integer, String> domains = domainService.fetchUserDomainsByIds(yandexuid.toString(), ids);

        return ids.stream().collect(Collectors.toMap(
                String::valueOf,
                domainId -> domains.getOrDefault(domainId, "")
        ));
    }

    @Path("lilucrm")
    public Class<LiluCrmResource> liluCrm() {
        return LiluCrmResource.class;
    }

    @Path("cdp")
    public Class<CdpResource> cdp() {
        return CdpResource.class;
    }

    @Path("direct/ca")
    public Class<DirectCaResource> direct_ca() {
        return DirectCaResource.class;
    }
}
