package ru.yandex.crypta.api.rest.resource.ext.lilucrm;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import io.swagger.annotations.ApiOperation;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.VultureGraphService;
import ru.yandex.crypta.idm.Roles;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
@RolesAllowed({"2000609", "2000613", "2000611", Roles.ADMIN})
public class LiluCrmResource {

    @Inject
    private VultureGraphService vultureGraphService;

    private Optional<Graph> getGraphFromVulture(GraphId graphId) {
        return vultureGraphService.getById(graphId);
    }

    @GET
    @Path("graph")
    @ApiOperation("Retrieve identifiers graph for LiluCRM")
    public List<LiluCrmGraph> getGraphForLiluCrm(
            @QueryParam("yandexuid") String yandexuid,
            @QueryParam("puid") String puid
    ) {
        var graphIds = Map.of("yandexuid", yandexuid != null ? yandexuid : "",
                "puid", puid != null ? puid : "")
                .entrySet().stream()
                .filter(e -> !e.getValue().isEmpty())
                .map(e -> new GraphId(e.getValue(), e.getKey()))
                .collect(Collectors.toList());

        var graphs = graphIds.stream()
                .map(this::getGraphFromVulture)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .map(Graph::getGraphComponents)
                .map(l -> l.get(0))
                .map(LiluCrmResource::convertGraph)
                .collect(Collectors.groupingBy(LiluCrmGraph::getCryptaId))
                .values()
                .stream()
                .map(l -> l.get(0))
                .collect(Collectors.toList());

        if (graphs.isEmpty()) {
            throw Exceptions.notFound();
        }

        return graphs;
    }

    private static LiluCrmGraph.Node convertVertex(Vertex v) {
        return new LiluCrmGraph.Node(v.getIdValue(), v.getIdType());
    }

    private static LiluCrmGraph.Edge convertEdge(Edge e, List<Vertex> vertices) {
        int vertex1 = vertices.indexOf(e.getVertex1());
        int vertex2 = vertices.indexOf(e.getVertex2());
        var attributes = Map.of(
                "source_type", e.getSourceType(),
                "log_source", e.getLogSource()
        );
        return new LiluCrmGraph.Edge(vertex1, vertex2, attributes);
    }

    private static LiluCrmGraph convertGraph(GraphComponent graph) {
        var graphVertices = graph.getVertices();
        var vertices = graphVertices
                .stream()
                .map(LiluCrmResource::convertVertex)
                .collect(Collectors.toList());
        var edges = graph.getEdges().stream()
                .map(edge -> convertEdge(edge, graphVertices))
                .collect(Collectors.toList());
        return new LiluCrmGraph(graph.getCryptaId(), vertices, edges);
    }
}
