package ru.yandex.crypta.api.rest.resource.graph;

import java.util.Optional;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.jgrapht.io.ExportException;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.graph.api.utils.GraphDiffHelper;
import ru.yandex.crypta.graph.api.utils.GraphvizExporter;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.idm.Roles.Portal;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;

@Path("graph")
@Api(tags = {"graph"})
@Produces(JsonUtf8.MEDIA_TYPE)
@RolesAllowed({Portal.GRAPH, Portal.GRAPH_PRIVATE, Portal.GRAPH_ANTIFRAUD})
public class GraphResource {

    private static final InfoParams INCLUDE_ALL_INFO = new InfoParams(true, true, true, true);
    private static final InfoParams INCLUDE_NO_INFO = new InfoParams(false, false, false, false);

    private final GraphServices services;

    @Inject
    public GraphResource(GraphServices services) {
        this.services = services;
    }

    @GET
    @ApiOperation(value = "Gets graph by any id")
    public Graph getById(
            @QueryParam("id") @NotNull String id,
            @QueryParam("idType") @NotNull String idType,
            @QueryParam("matchingType") @NotNull String matchingType,
            @QueryParam("matchingScope") @NotNull String matchingScope,
            @QueryParam("depth") @DefaultValue("0") String depth,
            @QueryParam("includeInfo") @DefaultValue("true") boolean includeInfo) {

        Integer soupDepth;
        if (depth.equals("")) {
            soupDepth = 0;
        } else {
            soupDepth = Integer.valueOf(depth);
        }

        return services.getByType(matchingType).getById(
                new GraphId(normalizeId(id, idType), idType),
                new SearchParams(null, matchingType, matchingScope, soupDepth),
                INCLUDE_ALL_INFO
        ).orElseThrow(Exceptions::notFound);
    }

    private String normalizeId(String id, String idType) {
        EIdType eIdType = Soup.CONFIG.getIdTypes().nameToType(idType);
        if (eIdType != null) {
            return new Identifier(idType, id).getNormalizedValue();
        } else {
            return id;
        }
    }

    @GET
    @Path("dot")
    @ApiOperation(value = "Export graph in .dot format")
    @Produces("text/plain; charset=utf-8")
    public String exportGraph(
            @QueryParam("id") @NotNull String id,
            @QueryParam("idType") @NotNull String idType,
            @QueryParam("matchingType") @NotNull String matchingType,
            @QueryParam("matchingScope") @NotNull String matchingScope,
            @QueryParam("depth") int depth
    ) {
        Optional<Graph> maybeGraph = services.getByType(matchingType).getById(
                new GraphId(id, idType),
                new SearchParams(null, matchingType, matchingScope, depth),
                INCLUDE_NO_INFO
        );

        return maybeGraph.map(
                g -> {
                    try {
                        return GraphvizExporter.exportGraphToDot(g, false);
                    } catch (ExportException e) {
                        throw Exceptions.unchecked(e);
                    }
                }
        ).orElse("");
    }

    @GET
    @Path("diff/dot")
    @ApiOperation(value = "Export graphs diff in .dot format")
    @Produces("text/plain; charset=utf-8")
    public String exportGraphsDiff(@QueryParam("id1") @NotNull String id1,
                                   @QueryParam("idType1") @NotNull String idType1,
                                   @QueryParam("matchingType1") @NotNull String matchingType1,
                                   @QueryParam("matchingScope1") @NotNull String matchingScope1,
                                   @QueryParam("depth1") int depth1,
                                   @QueryParam("id2") @NotNull String id2,
                                   @QueryParam("idType2") @NotNull String idType2,
                                   @QueryParam("matchingType2") @NotNull String matchingType2,
                                   @QueryParam("matchingScope2") @NotNull String matchingScope2,
                                   @QueryParam("depth2") int depth2
    ) {
        SearchParams searchParams1 = new SearchParams(null, matchingType1, matchingScope1, depth1);
        Graph graph1 = services.getByType(matchingType1)
                .getById(new GraphId(id1, idType1), searchParams1, INCLUDE_NO_INFO)
                .orElseThrow(Exceptions::notFound);

        SearchParams searchParams2 = new SearchParams(null, matchingType2, matchingScope2, depth2);
        Graph graph2 = services.getByType(matchingType2)
                .getById(new GraphId(id2, idType2), searchParams2, INCLUDE_NO_INFO)
                .orElseThrow(Exceptions::notFound);

        Graph diff = GraphDiffHelper.getGraphsDiff(graph1, graph2);
        try {
            return GraphvizExporter.exportGraphToDot(diff, true);
        } catch (ExportException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @GET
    @Path("diff")
    @ApiOperation(value = "Gets diff of two graphs")
    public Graph getGraphsDiff(@QueryParam("id1") @NotNull String id1,
                               @QueryParam("idType1") @NotNull String idType1,
                               @QueryParam("matchingType1") @NotNull String matchingType1,
                               @QueryParam("matchingScope1") @NotNull String matchingScope1,
                               @QueryParam("depth1") int depth1,
                               @QueryParam("id2") @NotNull String id2,
                               @QueryParam("idType2") @NotNull String idType2,
                               @QueryParam("matchingType2") @NotNull String matchingType2,
                               @QueryParam("matchingScope2") @NotNull String matchingScope2,
                               @QueryParam("depth2") int depth2
    ) {
        SearchParams searchParams1 = new SearchParams(null, matchingType1, matchingScope1, depth1);
        Graph graph1 = services.getByType(matchingType1)
                .getById(new GraphId(id1, idType1), searchParams1, INCLUDE_ALL_INFO)
                .orElseThrow(Exceptions::notFound);
        SearchParams searchParams2 = new SearchParams(null, matchingType2, matchingScope2, depth2);
        Graph graph2 = services.getByType(matchingType2)
                .getById(new GraphId(id2, idType2), searchParams2, INCLUDE_ALL_INFO)
                .orElseThrow(Exceptions::notFound);

        return GraphDiffHelper.getGraphsDiff(graph1, graph2);
    }

    @Path("settings")
    public Class<GraphSettingsResource> settings() {
        return GraphSettingsResource.class;
    }

    @Path("ids")
    public Class<IdsStorageResource> ids() {
        return IdsStorageResource.class;
    }
}
