package ru.yandex.crypta.api.rest.resource.lab;

import java.util.List;
import java.util.Optional;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import io.swagger.annotations.ApiOperation;

import ru.yandex.crypta.api.rest.resource.lab.grab.AudienceGrab;
import ru.yandex.crypta.api.rest.resource.lab.grab.AudienceGrabResource;
import ru.yandex.crypta.api.rest.resource.lab.grab.AudienceGrabTable;
import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.Audience;


@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class AudienceResource extends CommonLabResource {
    private final AudienceGrabTable table;

    @Inject
    public AudienceResource(LabService lab, PostgresClient postgresClient) {
        super(lab);
        this.table = new AudienceGrabTable(postgresClient.getJooqConfiguration());
    }

    @GET
    @ApiOperation(value = "Retrieve my audiences")
    public List<Audience> getMyAudiences() {
        return lab().audiences().getMyLabAudiences();
    }

    @GET
    @Path("{id}")
    @ApiOperation(value = "Retrieve audience by id")
    public Audience getAudience(
            @PathParam("id") @NotNull String id)
    {
        return lab().audiences().getLabAudienceById(id);
    }

    @POST
    @ApiOperation(value = "Create audience")
    public Audience createAudience(
            @QueryParam("name") @NotNull String name,
            @QueryParam("purpose") @NotNull String purpose,
            @QueryParam("login") @NotNull String login,
            @QueryParam("sourcePath") @NotNull String sourcePath,
            @QueryParam("sourceField") @NotNull String sourceField,
            @QueryParam("grabId") @NotNull String grabId
    )
    {
        Optional<AudienceGrab> grab = table.selectByIdQuery(grabId).fetchOptionalInto(AudienceGrab.class);
        if (!grab.isPresent() || !grab.get().getLogin().equals(login) || grab.get().hasExpired(60 * 5) ||
                !grab.get().getAuth())
        {
            throw Exceptions.notAuthenticated("Not authenticated at yandex.ru either login is incorrect");
        }

        return lab().audiences().createAudience(
                Audience.newBuilder()
                        .setName(name)
                        .setPurpose(purpose)
                        .setLogin(login)
                        .setSourcePath(sourcePath)
                        .setSourceField(sourceField)
        );
    }


    @PUT
    @Path("{id}/modify")
    @ApiOperation(value = "Modify audience")
    public Audience modifyAudience(
            @PathParam("id") @NotNull String id,
            @QueryParam("sourcePath") String sourcePath,
            @QueryParam("sourceField") String sourceField
    )
    {
        return lab().audiences().modifyAudience(id, sourcePath, sourceField);
    }

    @PUT
    @Path("{id}/set/external_id")
    @ApiOperation(value = "Set audience externalId, i.e. id from Ya.Audience")
    @RolesAllowed({Roles.ADMIN})
    public Audience setExternalId(
            @PathParam("id") @NotNull String id,
            @QueryParam("externalId") @NotNull Long audienceId
    )
    {
        return lab().audiences().setExternalId(id, audienceId);
    }

    @PUT
    @Path("{id}/state")
    @ApiOperation(value = "Update audience state")
    public Audience updateAudienceState(
            @PathParam("id") @NotNull String id,
            @QueryParam("state") @NotNull Audience.State state
    )
    {
        return lab().audiences().updateState(id, state);
    }

    @DELETE
    @Path("{id}/delete")
    @ApiOperation(value = "Delete audience from Ya.Audience")
    public Audience deleteAudience(@PathParam("id") @NotNull String id) {
        return lab().audiences().deleteAudience(id);
    }


    @DELETE
    @Path("{id}")
    @ApiOperation(value = "Delete audience from Lab.Crypta")
    public Audience deleteLabAudience(@PathParam("id") @NotNull String id) {
        return lab().audiences().deleteLabAudience(id);
    }

    @Path("grab")
    @ApiOperation(value = "Operate with user marks")
    public Class<AudienceGrabResource> grab() {
        return AudienceGrabResource.class;
    }
}
