package ru.yandex.crypta.api.rest.resource.lab;

import java.util.List;
import java.util.Optional;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.Rule;
import ru.yandex.crypta.lab.proto.RuleCondition;
import ru.yandex.crypta.lab.proto.RuleEstimateCoverage;
import ru.yandex.crypta.lab.proto.TSimpleSampleStatsWithInfo;


@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class ConstructorResource extends CommonLabResource {

    @Inject
    public ConstructorResource(LabService lab) {
        super(lab);
    }

    @GET
    @Path("rule")
    @ApiOperation(value = "Get all rules")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public List<Rule> getAllRules() {
        return lab().constructor().getAllRules();
    }

    @GET
    @Path("rule/condition")
    @ApiOperation(value = "Get all rules conditions")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public List<RuleCondition> getAllRulesConditions() {
        return lab().constructor().getAllRulesConditions();
    }

    @GET
    @Path("rule/{source}/condition")
    @ApiOperation(value = "Get all rules conditions by source")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public List<RuleCondition> getAllRulesConditionsBySource(
            @PathParam("source") RuleCondition.Source source
    )
    {
        return lab().constructor().getAllRulesConditionsBySource(source);
    }

    @POST
    @Path("rule")
    @ApiOperation(value = "Create rule")
    public Rule createRule(
            @QueryParam("name") @ApiParam("Name of the rule") @NotNull String name,
            @QueryParam("days") @ApiParam("Number of days to be processed") Long days,
            @QueryParam("min_days") @ApiParam("Minimum number of days of activity") Long minDays
    )
    {
        Rule.Builder prototype = Rule.newBuilder()
                .setName(name)
                .setDays(Optional.ofNullable(days).orElse(35L))
                .setMinDays(Optional.ofNullable(minDays).orElse(2L));
        return lab().constructor().createRule(prototype);
    }

    @POST
    @Path("export/{id}/rule")
    @ApiOperation(value = "Create rule for export")
    public Rule createExportRule(
            @PathParam("id") @ApiParam("Export ID") @NotNull String exportId,
            @QueryParam("name") @ApiParam("Name of the rule") @NotNull String name,
            @QueryParam("days") @ApiParam("Number of days to be processed") Long days,
            @QueryParam("min_days") @ApiParam("Minimum number of days of activity") Long minDays
    )
    {
        Rule.Builder prototype = Rule.newBuilder()
                .setName(name)
                .setDays(Optional.ofNullable(days).orElse(35L))
                .setMinDays(Optional.ofNullable(minDays).orElse(2L));
        return lab().constructor().createExportRule(exportId, prototype);
    }

    @GET
    @Path("rule/{id}")
    @ApiOperation(value = "Retrieve rule")
    public Rule getRule(@PathParam("id") @NotNull String id) {
        return lab().constructor().getRule(id);
    }

    @PUT
    @Path("rule/{id}")
    @ApiOperation(value = "Update rule")
    public Rule updateRule(
            @PathParam("id") @NotNull String id,
            @QueryParam("name") String name,
            @QueryParam("days") Long days,
            @QueryParam("min_days") Long minDays

    )
    {
        Rule.Builder prototype = Rule.newBuilder();
        Optional.ofNullable(name).ifPresent(prototype::setName);
        Optional.ofNullable(days).ifPresent(prototype::setDays);
        Optional.ofNullable(minDays).ifPresent(prototype::setMinDays);
        return lab().constructor().updateRule(id, prototype);
    }

    @DELETE
    @Path("rule/{id}")
    @ApiOperation(value = "Delete rule")
    public Rule deleteRule(@PathParam("id") @NotNull String id) {
        return lab().constructor().deleteRule(id);
    }

    @DELETE
    @Path("export/{id}/rule")
    @ApiOperation(value = "Delete export rule")
    public Rule deleteExportRule(@PathParam("id") @ApiParam("Export ID") @NotNull String exportId) {
        return lab().constructor().deleteExportRule(exportId);
    }

    @POST
    @Path("rule/{id}/condition")
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    @ApiOperation(value = "Create rule condition")
    @Deprecated
    public RuleCondition createRuleCondition(
            @PathParam("id") @NotNull String ruleId,
            @FormParam("source") @NotNull RuleCondition.Source source,
            @FormParam("values") @NotNull List<String> values
    )
    {
        return lab().constructor().putRuleCondition(ruleId, source, values);
    }

    @POST
    @Path("rule/{id}/condition/{source}")
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    @ApiOperation(value = "Put rule condition")
    public RuleCondition putRuleCondition(
            @PathParam("id") @NotNull String ruleId,
            @PathParam("source") @NotNull RuleCondition.Source source,
            @FormParam("values") @NotNull List<String> values
    )
    {
        return lab().constructor().putRuleCondition(ruleId, source, values);
    }

    @GET
    @Path("rule/{id}/condition/{source}/{state}")
    @ApiOperation(value = "Retrieve rule condition")
    public RuleCondition getRuleCondition(
            @PathParam("id") @NotNull String ruleId,
            @PathParam("source") @NotNull RuleCondition.Source source,
            @PathParam("state") @NotNull RuleCondition.State state
    )
    {
        return lab().constructor().getRuleCondition(ruleId, source, state);
    }

    @GET
    @Path("rule/{id}/condition/{source}")
    @ApiOperation(value = "Retrieve rule conditions")
    public List<RuleCondition> getRuleConditionBySource(
            @PathParam("id") @NotNull String ruleId,
            @PathParam("source") @NotNull RuleCondition.Source source
    )
    {
        return lab().constructor().getRuleConditionBySource(ruleId, source);
    }

    @GET
    @Path("rule/condition/{revision}")
    @ApiOperation(value = "Retrieve rule condition")
    public RuleCondition getRuleConditionByRevision(
            @PathParam("revision") @NotNull Long revision
    )
    {
        return lab().constructor().getRuleConditionByRevision(revision);
    }

    @PUT
    @Path("rule/{id}/condition/{source}")
    @ApiOperation(value = "Approve rule condition")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public RuleCondition approveRuleCondition(
            @PathParam("id") @NotNull String ruleId,
            @PathParam("source") @NotNull RuleCondition.Source source
    )
    {
        return lab().constructor().approveRuleCondition(ruleId, source);
    }

    @DELETE
    @Path("rule/{id}/condition/{source}/{state}")
    @ApiOperation(value = "Delete rule condition")
    public RuleCondition deleteRuleCondition(
            @PathParam("id") @NotNull String ruleId,
            @PathParam("source") @NotNull RuleCondition.Source source,
            @PathParam("state") @NotNull RuleCondition.State state
    )
    {
        return lab().constructor().deleteRuleCondition(ruleId, source, state);
    }

    @DELETE
    @Path("rule/{id}/condition/{source}")
    @ApiOperation(value = "Delete rule conditions of all states")
    public List<RuleCondition> deleteRuleConditionBySource(
            @PathParam("id") @NotNull String ruleId,
            @PathParam("source") @NotNull RuleCondition.Source source
    )
    {
        return lab().constructor().deleteRuleConditionBySource(ruleId, source);
    }

    @GET
    @Path("rule/condition/estimate/stats/{revision}")
    @ApiOperation(value = "Retrieve rule condition estimate stats")
    public TSimpleSampleStatsWithInfo getRuleConditionEstimateStatsByRevision(
            @PathParam("revision") @NotNull Long revision
    )
    {
        var estimate = lab().constructor().getRuleConditionEstimateStatsByRevision(revision);
        if (estimate.isEmpty()) {
            throw Exceptions.notFound("Rule estimates not found: " + revision);
        }

        var stats = lab().getStatsFromSiberia(Long.toUnsignedString(estimate.get().getUserSetId()));
        if (stats.isEmpty()) {
            throw Exceptions.notFound("Siberia stats not found: " + revision);
        }

        return stats.get();
    }

    @GET
    @Path("rule/condition/estimate/coverage/{revision}")
    @ApiOperation(value = "Retrieve rule condition estimate coverage")
    public RuleEstimateCoverage getRuleConditionEstimateCoverageByRevision(
            @PathParam("revision") @NotNull Long revision
    )
    {
        var estimate = lab().constructor().getRuleConditionEstimateStatsByRevision(revision);
        if (estimate.isEmpty()) {
            throw Exceptions.notFound("Rule estimates not found: " + revision);
        }

        return RuleEstimateCoverage.newBuilder()
                .setCoverage(estimate.get().getCoverage())
                .setTimestamp(estimate.get().getTimestamp())
                .setIsReady(estimate.get().getIsReady())
                .build();
    }

    @GET
    @Path("rule/estimate/stats/{rule_id}")
    @ApiOperation(value = "Retrieve rule estimate stats")
    public TSimpleSampleStatsWithInfo getRuleEstimateStats(
            @PathParam("rule_id") @NotNull String ruleId
    )
    {
        var estimate = lab().constructor().getRuleEstimateStats(ruleId);
        if (estimate.isEmpty()) {
            throw Exceptions.notFound("Rule estimates not found: " + ruleId);
        }

        var stats = lab().getStatsFromSiberia(Long.toUnsignedString(estimate.get().getUserSetId()));
        if (stats.isEmpty()) {
            throw Exceptions.notFound("Siberia stats not found: " + ruleId);
        }

        return stats.get();
    }

    @GET
    @Path("rule/estimate/coverage/{rule_id}")
    @ApiOperation(value = "Retrieve rule estimate coverage")
    public RuleEstimateCoverage getRuleEstimateCoverage(
            @PathParam("rule_id") @NotNull String ruleId
    )
    {
        var estimate = lab().constructor().getRuleEstimateStats(ruleId);
        if (estimate.isEmpty()) {
            throw Exceptions.notFound("Rule estimates not found: " + ruleId);
        }

        return RuleEstimateCoverage.newBuilder()
                .setCoverage(estimate.get().getCoverage())
                .setTimestamp(estimate.get().getTimestamp())
                .setIsReady(estimate.get().getIsReady())
                .build();
    }

    @POST
    @Path("rule/estimate_stats/{rule_id}")
    @ApiOperation(value = "Update rule estimate stats")
    public String updateRuleEstimateStats(
            @PathParam("rule_id") @NotNull String ruleId
    )
    {
        if (!lab().constructor().updateRuleEstimateStats(ruleId)) {
            throw Exceptions.internal("Failed to update rule estimate stats");
        }
        return "Ok";
    }
}
