package ru.yandex.crypta.api.rest.resource.lab;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import com.fasterxml.jackson.databind.JsonNode;
import io.swagger.annotations.ApiOperation;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.LabService;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class ExtResource {

    private final LabService lab;

    private @QueryParam("lang")
    Language language = null;

    @Inject
    public ExtResource(LabService lab) {
        this.lab = lab;
    }

    private LabService lab() {
        return lab.withLanguage(Language.orDefault(language));
    }

    @GET
    @Path("market/segments")
    @ApiOperation("Retrieve list of segments for Market")
    @RolesAllowed(Roles.Lab.Ext.MARKET)
    public List<SimpleSegment> getSegmentsForMarket() {
        return lab().segments().getAll().stream().flatMap(eachSegment ->
                eachSegment.getExports().getExportsList().stream().map(eachExport -> {
                    long keywordId = eachExport.getKeywordId();
                    long segmentId = eachExport.getSegmentId();
                    String name = eachSegment.getName();
                    String description = eachSegment.getDescription();
                    return new SimpleSegment(keywordId, segmentId, name, description);
                })
        ).collect(Collectors.toList());
    }

    @GET
    @Path("yamoney/segments")
    @ApiOperation("Retrieve list of segments for Yandex.Money")
    @RolesAllowed(Roles.Lab.Ext.YAMONEY)
    public List<SimpleSegment> getSegmentsForYaMoney() {
        return getSegmentsForMarket();
    }

    @GET
    @Path("direct/taxonomy")
    @ApiOperation("Retrieve taxonomy of segments for Direct")
    @RolesAllowed(Roles.Lab.Ext.DIRECT)
    public JsonNode getTaxonomyForDirect() {
        return lab().getDirectTaxonomy();
    }

    @GET
    @Path("metrica/taxonomy")
    @ApiOperation("Retrieve taxonomy of segments for Metrica")
    @RolesAllowed({Roles.Lab.Ext.METRICA, "2010128", "2010130"})
    public JsonNode getTaxonomyForMetrica() {
        return lab().getDirectTaxonomy();
    }

    @GET
    @Path("mediaservices/segments_meta")
    @ApiOperation("Retrieves segments description for Mediaservices")
    @RolesAllowed(Roles.Lab.Ext.MEDIASERVICES)
    public List<Map<String, String>> getSegmentsMetaForMediaservices() {
        return lab().getSegmentsMetaForMediaservices();
    }

    private static class SimpleSegment {
        private final long keywordId;
        private final long segmentId;
        private final String name;
        private final String description;

        private SimpleSegment(long keywordId, long segmentId, String name, String description) {
            this.keywordId = keywordId;
            this.segmentId = segmentId;
            this.name = name;
            this.description = description;
        }

        public long getKeywordId() {
            return keywordId;
        }

        public long getSegmentId() {
            return segmentId;
        }

        public String getName() {
            return name;
        }

        public String getDescription() {
            return description;
        }
    }

}
