package ru.yandex.crypta.api.rest.resource.lab;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Optional;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.google.common.io.ByteStreams;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.glassfish.jersey.media.multipart.FormDataParam;

import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.Model;
import ru.yandex.crypta.lab.proto.ModelSegmentRelation;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.Thresholds;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class ModelResource {

    private final LabService lab;

    @Inject
    public ModelResource(LabService lab) {
        this.lab = lab;
    }

    private static byte[] getByteData(InputStream uploadedInputStream) {
        try {
            return ByteStreams.toByteArray(uploadedInputStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @GET
    @ApiOperation(value = "Get all models")
    public List<Model> getAllModels(
            @QueryParam("state") Model.State state,
            @QueryParam("segmentId") String segmentId
    )
    {
        List<Model> models = lab.models().getAll();
        if (state != null) {
            models.removeIf(model -> !model.getState().equals(state));
        }
        if (segmentId != null) {
            models.removeIf(model -> {
                for (ModelSegmentRelation relation : model.getSegmentsList()) {
                    if (relation.getSegmentId().equals(segmentId)) {
                        return false;
                    }
                }
                return true;
            });
        }
        return models;
    }

    @DELETE
    @Path("{id}")
    @ApiOperation(value = "Delete model")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Model deleteModel(@PathParam("id") String id) {
        return lab.models().delete(id);
    }

    @GET
    @Path("{id}")
    @ApiOperation(value = "Get model")
    public Model getModel(@PathParam("id") String id) {
        return lab.models().get(id);
    }

    @GET
    @Path("tag/{tag}")
    @ApiOperation(value = "Get model by tag")
    public Model getModelByTag(@PathParam("tag") String tag) {
        return lab.models().getByTag(tag);
    }

    @POST
    @Path("{id}/enable")
    @ApiOperation(value = "Enable model")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Model enableModel(@PathParam("id") String id) {
        Model.Builder model = getModel(id).toBuilder().setState(Model.State.ENABLED);
        return lab.models().update(model);
    }

    @POST
    @Path("{id}/disable")
    @ApiOperation(value = "Disable model")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Model disableModel(@PathParam("id") String id) {
        Model.Builder model = getModel(id).toBuilder().setState(Model.State.DISABLED);
        return lab.models().update(model);
    }

    @GET
    @Path("{id}/file")
    @Produces({MediaType.APPLICATION_OCTET_STREAM, JsonUtf8.MEDIA_TYPE})
    @ApiOperation(value = "Get model file")
    @ApiResponses(
            @ApiResponse(code = 200, message = "Model file", response = File.class)
    )
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Response getModelFile(@PathParam("id") String id) {
        byte[] stream = lab.models().getFile(id);
        return Response
                .ok(stream, MediaType.APPLICATION_OCTET_STREAM_TYPE)
                .header("Content-Disposition", "attachment; filename=\"" + id + "\"")
                .build();
    }

    @GET
    @Path("tag/{tag}/file")
    @Produces({MediaType.APPLICATION_OCTET_STREAM, JsonUtf8.MEDIA_TYPE})
    @ApiOperation(value = "Get model file by tag")
    @ApiResponses(
            @ApiResponse(code = 200, message = "Model file by tag", response = File.class)
    )
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Response getModelFileByTag(@PathParam("tag") String tag) {
        byte[] stream = lab.models().getFileByTag(tag);
        return Response
                .ok(stream, MediaType.APPLICATION_OCTET_STREAM_TYPE)
                .header("Content-Disposition", "attachment; filename=\"" + tag + "\"")
                .build();
    }

    @PUT
    @Path("{id}/tag")
    @ApiOperation(value = "Set tag to model")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Model setTag(
            @PathParam("id") @NotNull String modelId,
            @QueryParam("tag") @NotNull String tag
    ) throws NotFoundException
    {
        Model.Builder model = getModel(modelId).toBuilder().setTag(tag);
        return lab.models().update(model);
    }

    @POST
    @Consumes(MediaType.MULTIPART_FORM_DATA)
    @ApiOperation(value = "Upload new model")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "file", dataType = "java.io.File", paramType = "form")
    })
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Model createModel(
            @QueryParam("description") @NotNull String description,
            @ApiParam(hidden = true) @FormDataParam("file") InputStream uploadedInputStream,
            @ApiParam(hidden = true) @FormDataParam("file") FormDataContentDisposition fileDetail
    ) throws NotFoundException
    {
        byte[] data = getByteData(uploadedInputStream);

        Model.Builder prototype = Model
                .newBuilder()
                .setDescription(description)
                .setState(Model.State.DISABLED);

        return lab.models().put(data, prototype);
    }

    @POST
    @Path("{id}/link")
    @ApiOperation(value = "Link model to segment")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Model linkModel(
            @PathParam("id") @NotNull String modelId,
            @QueryParam("segment_id") @NotNull String segmentId,
            @QueryParam("low_threshold") Double lowThreshold,
            @QueryParam("high_threshold") Double highThreshold
    )
    {
        Model model = getModel(modelId);
        Segment segment = lab.segments().get(segmentId);
        Thresholds.Builder thresholds = Thresholds.newBuilder();
        thresholds.setLow(Optional.ofNullable(lowThreshold).orElse(0.0));
        thresholds.setHigh(Optional.ofNullable(highThreshold).orElse(1.0));
        return lab.models().link(model, segment, thresholds.build());
    }

    @POST
    @Path("{id}/unlink")
    @ApiOperation(value = "Unlink model from segment")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Model unlinkModel(
            @PathParam("id") @NotNull String modelId,
            @QueryParam("segment_id") @NotNull String segmentId
    )
    {
        Model model = getModel(modelId);
        Segment segment = lab.segments().get(segmentId);
        return lab.models().unlink(model, segment);
    }
}
